#include "chip.h"


#ifdef ENABLE_ALL_CHIPS
    // Not on firmware; can read strings directly
    #define PSTR(s) s
#else
    // On firmware; use PSTR() macro from the AVR library
    #include <avr/pgmspace.h>
#endif


const char *chip_description(uint16_t chip_id) {

    switch (chip_id) {

#if defined ENABLE_CHIP_ADT7410
        case USBTENKI_CHIP_ADT7410:
            return PSTR("ADT7410 I2C Temperature sensor");
#endif
#if defined ENABLE_CHIP_SE95
        case USBTENKI_CHIP_SE95:
            return PSTR("SE95 I2C Temperature sensor");
#endif
#if defined ENABLE_CHIP_MCP9800
        case USBTENKI_CHIP_MCP9800:
            return PSTR("MCP980x I2C Temperature sensor");
#endif
#if defined ENABLE_CHIP_LM75
        case USBTENKI_CHIP_LM75:
            return PSTR("LM75 I2C Temperature sensor");
#endif
#if defined ENABLE_CHIP_LM92
        case USBTENKI_CHIP_LM92:
            return PSTR("LM92 I2C Temperature sensor");
#endif
#if defined ENABLE_CHIP_SHT_TEMP
        case USBTENKI_CHIP_SHT_TEMP:
            return PSTR("Sensirion SHT1x/7x Temperature");
#endif
#if defined ENABLE_CHIP_SHT_RH
        case USBTENKI_CHIP_SHT_RH:
            return PSTR("Sensirion SHT1x/7x Relative Humidity");
#endif
#if defined ENABLE_CHIP_BS02_TEMP
        case USBTENKI_CHIP_BS02_TEMP:
            return PSTR("BS02 Temperature");
#endif
#if defined ENABLE_CHIP_BS02_RH
        case USBTENKI_CHIP_BS02_RH:
            return PSTR("BS02 Relative Humidity");
#endif
#if defined ENABLE_CHIP_CC2_RH
        case USBTENKI_CHIP_CC2_RH:
            return PSTR("CC2 Relative Humidity");
#endif
#if defined ENABLE_CHIP_CC2_T
        case USBTENKI_CHIP_CC2_T:
            return PSTR("CC2 Temperature");
#endif
#if defined ENABLE_CHIP_SHT31_T
        case USBTENKI_CHIP_SHT31_T:
            return PSTR("SHT31 Temperature");
#endif
#if defined ENABLE_CHIP_SHT31_RH
        case USBTENKI_CHIP_SHT31_RH:
            return PSTR("SHT31 Relative Humidity");
#endif
#if defined ENABLE_CHIP_SHT35_T
        case USBTENKI_CHIP_SHT35_T:
            return PSTR("SHT35 Temperature");
#endif
#if defined ENABLE_CHIP_SHT35_RH
        case USBTENKI_CHIP_SHT35_RH:
            return PSTR("SHT35 Relative Humidity");
#endif
#if defined ENABLE_CHIP_SCD30_T
        case USBTENKI_CHIP_SCD30_T:
            return PSTR("SCD30 Temperature");
#endif
#if defined ENABLE_CHIP_SCD30_RH
        case USBTENKI_CHIP_SCD30_RH:
            return PSTR("SCD30 Relative Humidity");
#endif
#if defined ENABLE_CHIP_SHT31_T_INTERNAL
        case USBTENKI_CHIP_SHT31_T_INTERNAL:
            return PSTR("Internal Temperature");
#endif
#if defined ENABLE_CHIP_SHT31_RH_INTERNAL
        case USBTENKI_CHIP_SHT31_RH_INTERNAL:
            return PSTR("Internal Relative Humidity");
#endif
#if defined ENABLE_CHIP_SHT31_T_EXTERNAL
        case USBTENKI_CHIP_SHT31_T_EXTERNAL:
            return PSTR("External Temperature");
#endif
#if defined ENABLE_CHIP_SHT31_RH_EXTERNAL
        case USBTENKI_CHIP_SHT31_RH_EXTERNAL:
            return PSTR("External Relative Humidity");
#endif
#if defined ENABLE_CHIP_CCS811_TVOC
        case USBTENKI_CHIP_CCS811_TVOC:
            return PSTR("CCS811 TVOC PPB");
#endif
#if defined ENABLE_CHIP_CCS811_eCO2
        case USBTENKI_CHIP_CCS811_eCO2:
            return PSTR("CCS811 eCO2 PPM");
#endif
#if defined ENABLE_CHIP_SCD30_CO2
        case USBTENKI_CHIP_SCD30_CO2:
            return PSTR("SCD30 CO2 GAS PPM");
#endif
#if defined ENABLE_CHIP_SCD30_CO2_CAL
        case USBTENKI_CHIP_SCD30_CO2_CAL:
            return PSTR("SCD30 CO2 GAS PPM");
#endif
#if defined ENABLE_CHIP_CO2_PPM
        case USBTENKI_CHIP_CO2_PPM:
            return PSTR("CO2 GAS PPM");
#endif
#if defined ENABLE_CHIP_CO2_DXC200_FILTERED
        case USBTENKI_CHIP_CO2_DXC200_FILTERED:
            return PSTR("CO2 GAS PPM (filtered)");
#endif
#if defined ENABLE_CHIP_CO2_DXC200_INSTANT
        case USBTENKI_CHIP_CO2_DXC200_INSTANT:
            return PSTR("CO2 GAS PPM (instant)");
#endif
#if defined ENABLE_CHIP_SPS30_MC_PM1_0
        case USBTENKI_CHIP_SPS30_MC_PM1_0:
            return PSTR("Mass Concentration PM1.0");
#endif
#if defined ENABLE_CHIP_SPS30_MC_PM2_5
        case USBTENKI_CHIP_SPS30_MC_PM2_5:
            return PSTR("Mass Concentration PM2.5");
#endif
#if defined ENABLE_CHIP_SPS30_MC_PM4_0
        case USBTENKI_CHIP_SPS30_MC_PM4_0:
            return PSTR("Mass Concentration PM4.0");
#endif
#if defined ENABLE_CHIP_SPS30_MC_PM10
        case USBTENKI_CHIP_SPS30_MC_PM10:
            return PSTR("Mass Concentration PM10");
#endif
#if defined ENABLE_CHIP_SPS30_NC_PM0_5
        case USBTENKI_CHIP_SPS30_NC_PM0_5:
            return PSTR("Number Concentration PM0.5");
#endif
#if defined ENABLE_CHIP_SPS30_NC_PM1_0
        case USBTENKI_CHIP_SPS30_NC_PM1_0:
            return PSTR("Number Concentration PM1.0");
#endif
#if defined ENABLE_CHIP_SPS30_NC_PM2_5
        case USBTENKI_CHIP_SPS30_NC_PM2_5:
            return PSTR("Number Concentration PM2.5");
#endif
#if defined ENABLE_CHIP_SPS30_NC_PM4_0
        case USBTENKI_CHIP_SPS30_NC_PM4_0:
            return PSTR("Number Concentration PM4.0");
#endif
#if defined ENABLE_CHIP_SPS30_NC_PM10
        case USBTENKI_CHIP_SPS30_NC_PM10:
            return PSTR("Number Concentration PM10");
#endif
#if defined ENABLE_CHIP_SPS30_TYP_PART_SIZE
        case USBTENKI_CHIP_SPS30_TYP_PART_SIZE:
            return PSTR("Typical Particle Size");
#endif
#if defined ENABLE_CHIP_TSL2561_IR_VISIBLE
        case USBTENKI_CHIP_TSL2561_IR_VISIBLE:
            return PSTR("TSL2561 Channel 0 (IR+Visibile)");
#endif
#if defined ENABLE_CHIP_TSL2561_IR
        case USBTENKI_CHIP_TSL2561_IR:
            return PSTR("TSL2561 Channel 1 (IR only)");
#endif
#if defined ENABLE_CHIP_TSL2561_IR_VISIBLE_16X
        case USBTENKI_CHIP_TSL2561_IR_VISIBLE_16X:
            return PSTR("TSL2561 Channel 0 (IR+Visibile) 16X gain");
#endif
#if defined ENABLE_CHIP_TSL2561_IR_16X
        case USBTENKI_CHIP_TSL2561_IR_16X:
            return PSTR("TSL2561 Channel 1 (IR only) 16X gain");
#endif
#if defined ENABLE_CHIP_TSL2568_IR_VISIBLE
        case USBTENKI_CHIP_TSL2568_IR_VISIBLE:
            return PSTR("TSL2568 Channel 0 (IR+Visibile)");
#endif
#if defined ENABLE_CHIP_TSL2568_IR
        case USBTENKI_CHIP_TSL2568_IR:
            return PSTR("TSL2568 Channel 1 (IR only)");
#endif
#if defined ENABLE_CHIP_TSL2568_IR_VISIBLE_16X
        case USBTENKI_CHIP_TSL2568_IR_VISIBLE_16X:
            return PSTR("TSL2568 Channel 0 (IR+Visibile) 16X gain");
#endif
#if defined ENABLE_CHIP_TSL2568_IR_16X
        case USBTENKI_CHIP_TSL2568_IR_16X:
            return PSTR("TSL2568 Channel 1 (IR only)");
#endif
#if defined ENABLE_MCU_ADC0
        case USBTENKI_MCU_ADC0:
            return PSTR("Microcontroller ADC channel 0");
#endif
#if defined ENABLE_MCU_ADC1
        case USBTENKI_MCU_ADC1:
            return PSTR("Microcontroller ADC channel 1");
#endif
#if defined ENABLE_MCU_ADC2
        case USBTENKI_MCU_ADC2:
            return PSTR("Microcontroller ADC channel 2");
#endif
#if defined ENABLE_MCU_ADC3
        case USBTENKI_MCU_ADC3:
            return PSTR("Microcontroller ADC channel 3");
#endif
#if defined ENABLE_MCU_ADC4
        case USBTENKI_MCU_ADC4:
            return PSTR("Microcontroller ADC channel 4");
#endif
#if defined ENABLE_MCU_ADC5
        case USBTENKI_MCU_ADC5:
            return PSTR("Microcontroller ADC channel 5");
#endif
#if defined ENABLE_CHIP_MPXV7002
        case USBTENKI_CHIP_MPXV7002:
            return PSTR("MPXV7002 Differential air pressure sensor");
#endif
#if defined ENABLE_CHIP_MLH_A
        case USBTENKI_CHIP_MLH_A:
            return PSTR("MLH150-A Pressure sensor");
#endif
#if defined ENABLE_CHIP_MPX4115
        case USBTENKI_CHIP_MPX4115:
            return PSTR("MPX4115 Absolute air pressure sensor");
#endif
#if defined ENABLE_CHIP_MP3H6115A
        case USBTENKI_CHIP_MP3H6115A:
            return PSTR("MP3H6115A Absolute air pressure sensor");
#endif
#if defined ENABLE_CHIP_VOLTS
        case USBTENKI_CHIP_VOLTS:
            return PSTR("Ratiometric volts from ADC");
#endif
#if defined ENABLE_CHIP_VOLTS_REVERSE
        case USBTENKI_CHIP_VOLTS_REVERSE:
            return PSTR("Inverted ratiometric volts from ADC");
#endif
#if defined ENABLE_CHIP_DRACAL_EM1_BUS_VOLTAGE
        case USBTENKI_CHIP_DRACAL_EM1_BUS_VOLTAGE:
            return PSTR("Bus voltage");
#endif
#if defined ENABLE_CHIP_DRACAL_EM1_SHUNT_VOLTAGE
        case USBTENKI_CHIP_DRACAL_EM1_SHUNT_VOLTAGE:
            return PSTR("Shunt voltage");
#endif
#if defined ENABLE_CHIP_DRACAL_EM1_POWER
        case USBTENKI_CHIP_DRACAL_EM1_POWER:
            return PSTR("Power");
#endif
#if defined ENABLE_CHIP_DRACAL_EM1_CURRENT
        case USBTENKI_CHIP_DRACAL_EM1_CURRENT:
            return PSTR("Current");
#endif
#if defined ENABLE_CHIP_TACHOMETER
        case USBTENKI_CHIP_TACHOMETER:
            return PSTR("Tachometer");
#endif
#if defined ENABLE_CHIP_RTD300_PT100_2W
        case USBTENKI_CHIP_RTD300_PT100_2W:
            return PSTR("2-Wire PT100 Temperature sensor");
#endif
#if defined ENABLE_CHIP_RTD300_PT100_3W
        case USBTENKI_CHIP_RTD300_PT100_3W:
            return PSTR("3-Wire PT100 Temperature sensor");
#endif
#if defined ENABLE_CHIP_PT100_RTD
        case USBTENKI_CHIP_PT100_RTD:
            return PSTR("PT100 Temperature sensor");
#endif
#if defined ENABLE_CHIPX_PT100_TMP
        case USBTENKI_CHIPX_PT100_TMP:
            return PSTR("Temperature");
#endif
#if defined ENABLE_CHIP_TMC200_COLD || defined ENABLE_CHIP_THC_COLD
#if defined ENABLE_CHIP_TMC200_COLD
        case USBTENKI_CHIP_TMC200_COLD:
#endif
#if defined ENABLE_CHIP_THC_COLD
        case USBTENKI_CHIP_THC_COLD:
#endif
            return PSTR("Thermocouple cold junction temperature");
#endif
#if defined ENABLE_CHIP_THC_HOT
        case USBTENKI_CHIP_THC_HOT:
            return PSTR("Thermocouple hot junction temperature");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_K || defined ENABLE_CHIP_THC_TYPE_K
#if defined ENABLE_CHIP_TMC200_TYPE_K
        case USBTENKI_CHIP_TMC200_TYPE_K:
#endif
#if defined ENABLE_CHIP_THC_TYPE_K
        case USBTENKI_CHIP_THC_TYPE_K:
#endif
            return PSTR("Type-K Thermocouple");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_J || defined ENABLE_CHIP_THC_TYPE_J
#if defined ENABLE_CHIP_TMC200_TYPE_J
        case USBTENKI_CHIP_TMC200_TYPE_J:
#endif
#if defined ENABLE_CHIP_THC_TYPE_J
        case USBTENKI_CHIP_THC_TYPE_J:
#endif
            return PSTR("Type-J Thermocouple");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_T || defined ENABLE_CHIP_THC_TYPE_T
#if defined ENABLE_CHIP_TMC200_TYPE_T
        case USBTENKI_CHIP_TMC200_TYPE_T:
#endif
#if defined ENABLE_CHIP_THC_TYPE_T
        case USBTENKI_CHIP_THC_TYPE_T:
#endif
            return PSTR("Type-T Thermocouple");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_N || defined ENABLE_CHIP_THC_TYPE_N
#if defined ENABLE_CHIP_TMC200_TYPE_N
        case USBTENKI_CHIP_TMC200_TYPE_N:
#endif
#if defined ENABLE_CHIP_THC_TYPE_N
        case USBTENKI_CHIP_THC_TYPE_N:
#endif
            return PSTR("Type-N Thermocouple");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_S || defined ENABLE_CHIP_THC_TYPE_S
#if defined ENABLE_CHIP_TMC200_TYPE_S
        case USBTENKI_CHIP_TMC200_TYPE_S:
#endif
#if defined ENABLE_CHIP_THC_TYPE_S
        case USBTENKI_CHIP_THC_TYPE_S:
#endif
            return PSTR("Type-S Thermocouple");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_E || defined ENABLE_CHIP_THC_TYPE_E
#if defined ENABLE_CHIP_TMC200_TYPE_E
        case USBTENKI_CHIP_TMC200_TYPE_E:
#endif
        case USBTENKI_CHIP_THC_TYPE_E:
#if defined ENABLE_CHIP_THC_TYPE_E
            return PSTR("Type-E Thermocouple");
#endif
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_B || defined ENABLE_CHIP_THC_TYPE_B
#if defined ENABLE_CHIP_TMC200_TYPE_B
        case USBTENKI_CHIP_TMC200_TYPE_B:
#endif
#if defined ENABLE_CHIP_THC_TYPE_B
        case USBTENKI_CHIP_THC_TYPE_B:
#endif
            return PSTR("Type-B Thermocouple");
#endif
#if defined ENABLE_CHIP_TMC200_TYPE_R || defined ENABLE_CHIP_THC_TYPE_R
#if defined ENABLE_CHIP_TMC200_TYPE_R
        case USBTENKI_CHIP_TMC200_TYPE_R:
#endif
#if defined ENABLE_CHIP_THC_TYPE_R
        case USBTENKI_CHIP_THC_TYPE_R:
#endif
            return PSTR("Type-R Thermocouple");
#endif
#if defined ENABLE_CHIP_MLX90614_TA
        case USBTENKI_CHIP_MLX90614_TA:
            return PSTR("MLX90615 Ambient temperature");
#endif
#if defined ENABLE_CHIP_MLX90614_TOBJ
        case USBTENKI_CHIP_MLX90614_TOBJ:
            return PSTR("MLX90614 Object temperature");
#endif
#if defined ENABLE_CHIP_MS5611_P
        case USBTENKI_CHIP_MS5611_P:
            return PSTR("MS5611 Pressure");
#endif
#if defined ENABLE_CHIP_MS5611_T
        case USBTENKI_CHIP_MS5611_T:
            return PSTR("MS5611 Temperature");
#endif
#if defined ENABLE_CHIP_VEML6075_UVA
        case USBTENKI_CHIP_VEML6075_UVA:
            return PSTR("VEML6075 UVA");
#endif
#if defined ENABLE_CHIP_VEML6075_UVB
        case USBTENKI_CHIP_VEML6075_UVB:
            return PSTR("VEML6075 UVB");
#endif
#if defined ENABLE_CHIP_VEML6030_ALS
        case USBTENKI_CHIP_VEML6030_ALS:
            return PSTR("VEML6030 ALS");
#endif
#if defined ENABLE_CHIP_VEML6030_WHITE
        case USBTENKI_CHIP_VEML6030_WHITE:
            return PSTR("VEML6030 WHITE");
#endif
#if defined ENABLE_CHIP_NONE
        case USBTENKI_CHIP_NONE:
            return PSTR("Unused/unconfigured");
#endif
#if defined ENABLE_CHIP_RED
        case USBTENKI_CHIP_RED:
            return PSTR("Red light intensity");
#endif
#if defined ENABLE_CHIP_GREEN
        case USBTENKI_CHIP_GREEN:
            return PSTR("Green light intensity");
#endif
#if defined ENABLE_CHIP_BLUE
        case USBTENKI_CHIP_BLUE:
            return PSTR("Blue light intensity");
#endif
#if defined ENABLE_CHIP_IR
        case USBTENKI_CHIP_IR:
            return PSTR("IR light intensity");
#endif
#if defined ENABLE_VIRTUAL_HEXCOLOR
        case USBTENKI_VIRTUAL_HEXCOLOR:
            return PSTR("Hexadecimal color");
#endif
#if defined ENABLE_CHIPX_SHT3X_RH
        case USBTENKI_CHIPX_SHT3X_RH:
            return PSTR("Relative humidity");
#endif
#if defined ENABLE_CHIPX_SHT3X_T
        case USBTENKI_CHIPX_SHT3X_T:
            return PSTR("Temperature");
#endif
#if defined ENABLE_CHIPX_MS5611_P
        case USBTENKI_CHIPX_MS5611_P:
            return PSTR("Atmospheric pressure");
#endif
#if defined ENABLE_CHIPX_MS5611_T
        case USBTENKI_CHIPX_MS5611_T:
            return PSTR("Temperature");
#endif
#if defined ENABLE_CHIP_TICKER
        case USBTENKI_CHIP_TICKER:
            return PSTR("Ticker");
#endif
#if defined ENABLE_CHIP_MATH
        case USBTENKI_CHIP_MATH:
            return PSTR("Math");
#endif
#if defined ENABLE_VIRTUAL_INTERNAL_DEW_POINT
        case USBTENKI_VIRTUAL_INTERNAL_DEW_POINT:
            return PSTR("Internal Dew point");
#endif

#if defined ENABLE_VIRTUAL_DEW_POINT
        case USBTENKI_VIRTUAL_DEW_POINT:
            return PSTR("Dew point");
#endif

#if defined ENABLE_VIRTUAL_HUMIDEX
        case USBTENKI_VIRTUAL_HUMIDEX:
            return PSTR("Humidex");
#endif

#if defined ENABLE_VIRTUAL_HEAT_INDEX
        case USBTENKI_VIRTUAL_HEAT_INDEX:
            return PSTR("Heat index");
#endif

#if defined ENABLE_VIRTUAL_SHT75_COMPENSATED_RH
        case USBTENKI_VIRTUAL_SHT75_COMPENSATED_RH:
            return PSTR("Relative Humidity (Temp. compensated)");
#endif

#if defined ENABLE_VIRTUAL_TSL2561_LUX
        case USBTENKI_VIRTUAL_TSL2561_LUX:
            return PSTR("TSL2561 Lux");
#endif

#if defined ENABLE_VIRTUAL_TSL2568_LUX
        case USBTENKI_VIRTUAL_TSL2568_LUX:
             return PSTR("TSL2568 Lux");
#endif

#if defined ENABLE_VIRTUAL_ALTITUDE
        case USBTENKI_VIRTUAL_ALTITUDE:
            return PSTR("Altitude");
#endif

        default:
            return PSTR("unknown");

    }

}


#ifdef ENABLE_ALL_CHIPS

const char *chip_description_short(uint16_t chip_id) {

    switch(chip_id) {

        case USBTENKI_CHIP_ADT7410:
        case USBTENKI_CHIP_SE95:
        case USBTENKI_CHIP_MCP9800:
        case USBTENKI_CHIP_LM75:
        case USBTENKI_CHIP_LM92:
        case USBTENKI_CHIP_SHT_TEMP:
        case USBTENKI_CHIP_BS02_TEMP:
        case USBTENKI_CHIP_PT100_RTD:
        case USBTENKI_CHIPX_PT100_TMP:
        case USBTENKI_CHIP_TMC200_COLD:
        case USBTENKI_CHIP_TMC200_TYPE_K:
        case USBTENKI_CHIP_TMC200_TYPE_J:
        case USBTENKI_CHIP_TMC200_TYPE_N:
        case USBTENKI_CHIP_TMC200_TYPE_E:
        case USBTENKI_CHIP_TMC200_TYPE_T:
        case USBTENKI_CHIP_TMC200_TYPE_B:
        case USBTENKI_CHIP_TMC200_TYPE_S:
        case USBTENKI_CHIP_TMC200_TYPE_R:
        case USBTENKI_CHIP_RTD300_PT100_3W:
        case USBTENKI_CHIP_RTD300_PT100_2W:
        case USBTENKI_CHIP_MLX90614_TOBJ:
        case USBTENKI_CHIP_MLX90614_TA:
        case USBTENKI_CHIP_MS5611_T:
        case USBTENKI_CHIP_CC2_T:
        case USBTENKI_CHIP_SHT31_T:
        case USBTENKI_CHIP_SHT35_T:
        case USBTENKI_CHIP_THC_HOT:
        case USBTENKI_CHIP_THC_COLD:
        case USBTENKI_CHIP_THC_TYPE_K:
        case USBTENKI_CHIP_THC_TYPE_J:
        case USBTENKI_CHIP_THC_TYPE_T:
        case USBTENKI_CHIP_THC_TYPE_N:
        case USBTENKI_CHIP_THC_TYPE_S:
        case USBTENKI_CHIP_THC_TYPE_E:
        case USBTENKI_CHIP_THC_TYPE_B:
        case USBTENKI_CHIP_THC_TYPE_R:
        case USBTENKI_CHIP_SCD30_T:
        case USBTENKI_CHIP_SHT31_T_INTERNAL:
        case USBTENKI_CHIP_SHT31_T_EXTERNAL:
        case USBTENKI_CHIPX_SHT3X_T:
        case USBTENKI_CHIPX_MS5611_T:
            return "Temperature";

        case USBTENKI_CHIP_TSL2561_IR_VISIBLE:
        case USBTENKI_CHIP_TSL2568_IR_VISIBLE:
            return "Visible and IR";

        case USBTENKI_CHIP_TSL2561_IR_VISIBLE_16X:
        case USBTENKI_CHIP_TSL2568_IR_VISIBLE_16X:
            return "Visible and IR (16x gain)";

        case USBTENKI_CHIP_TSL2561_IR:
        case USBTENKI_CHIP_TSL2568_IR:
            return "IR";

        case USBTENKI_CHIP_TSL2561_IR_16X:
        case USBTENKI_CHIP_TSL2568_IR_16X:
            return "IR (16x gain)";

        case USBTENKI_CHIP_VEML6075_UVA:
            return "UVA";
        case USBTENKI_CHIP_VEML6075_UVB:
            return "UVB";
        case USBTENKI_CHIP_VEML6030_ALS:
            return "Ambient light";
        case USBTENKI_CHIP_VEML6030_WHITE:
            return "White light";

        case USBTENKI_VIRTUAL_SHT75_COMPENSATED_RH:
        case USBTENKI_CHIP_SHT_RH:
        case USBTENKI_CHIP_BS02_RH:
        case USBTENKI_CHIP_CC2_RH:
        case USBTENKI_CHIP_SHT31_RH:
        case USBTENKI_CHIP_SHT35_RH:
        case USBTENKI_CHIP_SCD30_RH:
        case USBTENKI_CHIP_SHT31_RH_INTERNAL:
        case USBTENKI_CHIP_SHT31_RH_EXTERNAL:
        case USBTENKI_CHIPX_SHT3X_RH:
            return "Relative Humidity";

        case USBTENKI_MCU_ADC0:
        case USBTENKI_MCU_ADC1:
        case USBTENKI_MCU_ADC2:
        case USBTENKI_MCU_ADC3:
        case USBTENKI_MCU_ADC4:
        case USBTENKI_MCU_ADC5:
            return "Raw ADC output";

        case USBTENKI_CHIP_MS5611_P:
        case USBTENKI_CHIP_MPXV7002:
        case USBTENKI_CHIP_MPX4115:
        case USBTENKI_CHIP_MP3H6115A:
        case USBTENKI_CHIP_MLH_A:
        case USBTENKI_CHIPX_MS5611_P:
            return "Pressure";

        case USBTENKI_CHIP_DRACAL_EM1_BUS_VOLTAGE:
        case USBTENKI_CHIP_DRACAL_EM1_SHUNT_VOLTAGE:
        case USBTENKI_CHIP_VOLTS_REVERSE:
        case USBTENKI_CHIP_VOLTS:
            return "Voltage";

        case USBTENKI_CHIP_DRACAL_EM1_POWER:
            return "Power";

        case USBTENKI_CHIP_DRACAL_EM1_CURRENT:
            return "Current";


        case USBTENKI_CHIP_D6F_V03A1:
            return "Air speed";

        case USBTENKI_CHIP_TACHOMETER:
            return "Frequency";

        case USBTENKI_CHIP_CO2_DXC200_FILTERED:
        case USBTENKI_CHIP_CO2_DXC200_INSTANT:
        case USBTENKI_CHIP_CO2_PPM:
        case USBTENKI_CHIP_SCD30_CO2:
        case USBTENKI_CHIP_SCD30_CO2_CAL:
            return "CO2 Gas PPM";

        case USBTENKI_CHIP_CCS811_eCO2:
            return "Gas PPM";

        case USBTENKI_CHIP_CCS811_TVOC:
            return "Gas PPB";

        case USBTENKI_CHIP_RED:
        case USBTENKI_CHIP_GREEN:
        case USBTENKI_CHIP_BLUE:
        case USBTENKI_CHIP_IR:
            return chip_description(chip_id);

        case USBTENKI_CHIP_SPS30_MC_PM1_0:
        case USBTENKI_CHIP_SPS30_MC_PM2_5:
        case USBTENKI_CHIP_SPS30_MC_PM4_0:
        case USBTENKI_CHIP_SPS30_MC_PM10:
        case USBTENKI_CHIP_SPS30_NC_PM0_5:
        case USBTENKI_CHIP_SPS30_NC_PM1_0:
        case USBTENKI_CHIP_SPS30_NC_PM2_5:
        case USBTENKI_CHIP_SPS30_NC_PM4_0:
        case USBTENKI_CHIP_SPS30_NC_PM10:
            return "Concentration";

        case USBTENKI_CHIP_SPS30_TYP_PART_SIZE:
            return "Size";

        case USBTENKI_VIRTUAL_DEW_POINT:
        case USBTENKI_VIRTUAL_INTERNAL_DEW_POINT:
        case USBTENKI_VIRTUAL_EXTERNAL_DEW_POINT:
            return "Dew point";

        case USBTENKI_VIRTUAL_HUMIDEX:
            return "Humidex";

        case USBTENKI_VIRTUAL_HEAT_INDEX:
            return "Heat index";

        case USBTENKI_VIRTUAL_TSL2568_LUX:
        case USBTENKI_VIRTUAL_TSL2561_LUX:
            return "Lux";

        case USBTENKI_VIRTUAL_ALTITUDE:
            return "Height";

        case USBTENKI_CHIP_NONE:
            return "N/A";

        case USBTENKI_VIRTUAL_HEXCOLOR:
            return "Color";

        case USBTENKI_CHIP_TICKER:
            return "Ticker";

        case USBTENKI_CHIP_MATH:
            return "Math";
    }

    return "unknown";

}

/**
 * Units index by chip ID (non-virtual)
 * Sparse array, but that's OK. The whole thing takes less than 1 kB.
 */
const unit_t CHIP_NATIVE_UNITS[] = {

    [USBTENKI_CHIP_MCP9800] = UNIT_CELSIUS,
    [USBTENKI_CHIP_ADT7410] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SE95] = UNIT_CELSIUS,
    [USBTENKI_CHIP_LM75] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SHT_TEMP] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SHT_RH] = UNIT_RH,
    [USBTENKI_CHIP_BS02_TEMP] = UNIT_CELSIUS,
    [USBTENKI_CHIP_BS02_RH] = UNIT_RH,
    [USBTENKI_CHIP_MPXV7002] = UNIT_KPA,
    [USBTENKI_CHIP_MP3H6115A] = UNIT_KPA,
    [USBTENKI_CHIP_MPX4115] = UNIT_KPA,
    [USBTENKI_CHIP_MLH_A] = UNIT_PSI,
    [USBTENKI_CHIP_TSL2561_IR_VISIBLE_16X] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2561_IR_16X] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2568_IR_VISIBLE_16X] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2568_IR_16X] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2561_IR_VISIBLE] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2561_IR] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2568_IR_VISIBLE] = UNIT_RAW,
    [USBTENKI_CHIP_TSL2568_IR] = UNIT_RAW,
    [USBTENKI_CHIP_D6F_V03A1] = UNIT_METER_SEC,
    [USBTENKI_CHIP_VOLTS_REVERSE] = UNIT_VOLT,
    [USBTENKI_CHIP_VOLTS] = UNIT_VOLT,
    [USBTENKI_CHIP_TACHOMETER] = UNIT_HZ,
    [USBTENKI_CHIP_PT100_RTD] = UNIT_CELSIUS,
    [USBTENKI_CHIPX_PT100_TMP] = UNIT_CELSIUS,
    [USBTENKI_CHIP_DRACAL_EM1_BUS_VOLTAGE] = UNIT_VOLT,
    [USBTENKI_CHIP_DRACAL_EM1_SHUNT_VOLTAGE] = UNIT_VOLT,
    [USBTENKI_CHIP_DRACAL_EM1_POWER] = UNIT_WATT,
    [USBTENKI_CHIP_DRACAL_EM1_CURRENT] = UNIT_AMP,
    [USBTENKI_CHIP_RTD300_PT1000_3W] = UNIT_CELSIUS,
    [USBTENKI_CHIP_RTD300_PT100_3W] = UNIT_CELSIUS,
    [USBTENKI_CHIP_RTD300_PT100_2W] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_COLD] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_K] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_J] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_N] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_E] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_B] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_S] = UNIT_CELSIUS,
    [USBTENKI_CHIP_TMC200_TYPE_R] = UNIT_CELSIUS,
    [USBTENKI_CHIP_CO2_DXC200_FILTERED] = UNIT_PERCENT,
    [USBTENKI_CHIP_CO2_DXC200_INSTANT] = UNIT_PERCENT,
    [USBTENKI_MCU_ADC0] = UNIT_CELSIUS,
    [USBTENKI_MCU_ADC1] = UNIT_CELSIUS,
    [USBTENKI_MCU_ADC2] = UNIT_CELSIUS,
    [USBTENKI_MCU_ADC3] = UNIT_CELSIUS,
    [USBTENKI_MCU_ADC4] = UNIT_CELSIUS,
    [USBTENKI_MCU_ADC5] = UNIT_CELSIUS,
    [USBTENKI_CHIP_MLX90614_TA] = UNIT_CELSIUS,
    [USBTENKI_CHIP_MLX90614_TOBJ] = UNIT_CELSIUS,
    [USBTENKI_CHIP_MS5611_P] = UNIT_PA,
    [USBTENKI_CHIP_MS5611_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_CC2_RH] = UNIT_RH,
    [USBTENKI_CHIP_CC2_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SHT31_RH_INTERNAL] = UNIT_RH,
    [USBTENKI_CHIP_SHT31_RH_EXTERNAL] = UNIT_RH,
    [USBTENKI_CHIP_SHT35_RH] = UNIT_RH,
    [USBTENKI_CHIP_SHT31_RH] = UNIT_RH,
    [USBTENKI_CHIP_SHT31_T_INTERNAL] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SHT31_T_EXTERNAL] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SHT35_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SHT31_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_CCS811_TVOC] = UNIT_PPB,
    [USBTENKI_CHIP_CCS811_eCO2] = UNIT_PPM,
    [USBTENKI_CHIP_CO2_PPM] = UNIT_PPM,
    [USBTENKI_CHIP_SCD30_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_SCD30_RH] = UNIT_RH,
    [USBTENKI_CHIP_SCD30_CO2] = UNIT_PPM,
    [USBTENKI_CHIP_SCD30_CO2_CAL] = UNIT_PPM,
    [USBTENKI_CHIP_SPS30_MC_PM1_0] = UNIT_uG_PER_M3,
    [USBTENKI_CHIP_SPS30_MC_PM2_5] = UNIT_uG_PER_M3,
    [USBTENKI_CHIP_SPS30_MC_PM4_0] = UNIT_uG_PER_M3,
    [USBTENKI_CHIP_SPS30_MC_PM10] = UNIT_uG_PER_M3,
    [USBTENKI_CHIP_SPS30_NC_PM0_5] = UNIT_COUNT_PER_CM3,
    [USBTENKI_CHIP_SPS30_NC_PM1_0] = UNIT_COUNT_PER_CM3,
    [USBTENKI_CHIP_SPS30_NC_PM2_5] = UNIT_COUNT_PER_CM3,
    [USBTENKI_CHIP_SPS30_NC_PM4_0] = UNIT_COUNT_PER_CM3,
    [USBTENKI_CHIP_SPS30_NC_PM10] = UNIT_COUNT_PER_CM3,
    [USBTENKI_CHIP_SPS30_TYP_PART_SIZE] = UNIT_MICROMETER,
    [USBTENKI_CHIP_THC_TYPE_K] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_J] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_T] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_N] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_S] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_E] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_B] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_TYPE_R] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_HOT] = UNIT_CELSIUS,
    [USBTENKI_CHIP_THC_COLD] = UNIT_CELSIUS,
    [USBTENKI_CHIP_RED] = UNIT_ARBITRARY,
    [USBTENKI_CHIP_GREEN] = UNIT_ARBITRARY,
    [USBTENKI_CHIP_BLUE] = UNIT_ARBITRARY,
    [USBTENKI_CHIP_IR] = UNIT_ARBITRARY,
    [USBTENKI_CHIP_VEML6030_ALS] = UNIT_LUX,
    [USBTENKI_CHIP_VEML6030_WHITE] = UNIT_ARBITRARY,
    [USBTENKI_CHIP_VEML6075_UVA] = UNIT_uW_PER_CM2,
    [USBTENKI_CHIP_VEML6075_UVB] = UNIT_uW_PER_CM2,
    [USBTENKI_CHIP_VEML6075_UVCOMP1] = UNIT_ARBITRARY,
    [USBTENKI_CHIP_VEML6075_UVCOMP2] = UNIT_ARBITRARY,
    [USBTENKI_VIRTUAL_DEW_POINT] = UNIT_CELSIUS,
    [USBTENKI_VIRTUAL_HUMIDEX] = UNIT_CELSIUS,
    [USBTENKI_VIRTUAL_HEAT_INDEX] = UNIT_CELSIUS,
    [USBTENKI_VIRTUAL_TSL2561_LUX] = UNIT_LUX,
    [USBTENKI_VIRTUAL_TSL2568_LUX] = UNIT_LUX,
    [USBTENKI_VIRTUAL_SHT75_COMPENSATED_RH] = UNIT_RH,
    [USBTENKI_VIRTUAL_ALTITUDE] = UNIT_METER,
    [USBTENKI_VIRTUAL_HEXCOLOR] = UNIT_HEXCOLOR,
    [USBTENKI_VIRTUAL_INTERNAL_DEW_POINT] = UNIT_CELSIUS,
    [USBTENKI_VIRTUAL_EXTERNAL_DEW_POINT] = UNIT_CELSIUS,
    [USBTENKI_CHIPX_SHT3X_RH] = UNIT_RH,
    [USBTENKI_CHIPX_SHT3X_T] = UNIT_CELSIUS,
    [USBTENKI_CHIPX_MS5611_P] = UNIT_PA,
    [USBTENKI_CHIPX_MS5611_T] = UNIT_CELSIUS,
};

const char *chip_error_to_string(int32_t error) {

    switch (error) {
        case CHIP_ERROR_TICKER_DEVICE:
            return "Tick";
        case CHIP_ERROR_UNKNOWN_CHIP:
            return "Unknown Chip";
        case CHIP_ERROR_NO_DATA:
            return "No Data";
        case CHIP_ERROR_SATURATED:
            return "Saturated";
        case CHIP_ERROR_SENSOR_ERROR:
            return "Sensor Error";
        case CHIP_ERROR_PROBE_DISCONNECTED:
            return "Probe Disconnected";
        case CHIP_ERROR_OUT_OF_RANGE:
            return "Out Of Range";
        case CHIP_ERROR_INVALID_DATA:
            return "Invalid Data";
        case CHIP_ERROR_SENSOR_SHORTED:
            return "Sensor Shorted";
    }

    return "Undefined Error";

}

const char *chip_error_to_string_no_spaces(int32_t error) {

    switch (error) {
        case CHIP_ERROR_TICKER_DEVICE:
            return "Tick";
        case CHIP_ERROR_UNKNOWN_CHIP:
            return "UnknownChip";
        case CHIP_ERROR_NO_DATA:
            return "NoData";
        case CHIP_ERROR_SATURATED:
            return "Saturated";
        case CHIP_ERROR_SENSOR_ERROR:
            return "SensorError";
        case CHIP_ERROR_PROBE_DISCONNECTED:
            return "ProbeDisconnected";
        case CHIP_ERROR_OUT_OF_RANGE:
            return "OutOfRange";
        case CHIP_ERROR_INVALID_DATA:
            return "InvalidData";
        case CHIP_ERROR_SENSOR_SHORTED:
            return "SensorShorted";
    }

    return "UndefinedError";

}

const char *chip_thermocouple_type_to_string(int8_t type) {

    switch (type)
    {
        case THERMOCOUPLE_TYPE_K: return "K";
        case THERMOCOUPLE_TYPE_J: return "J";
        case THERMOCOUPLE_TYPE_T: return "T";
        case THERMOCOUPLE_TYPE_N: return "N";
        case THERMOCOUPLE_TYPE_S: return "S";
        case THERMOCOUPLE_TYPE_E: return "E";
        case THERMOCOUPLE_TYPE_B: return "B";
        case THERMOCOUPLE_TYPE_R: return "R";
    }
    return "?";

}

int8_t chip_thermocouple_string_to_type(const char *type) {

    if (!type) {
        return -1;
    }

    if (type[0] == '\0' || type[1] != '\0') {  // strlen(type) == 1
        return -1;
    }

    switch(type[0]) {
        case 'k':
        case 'K':
            return THERMOCOUPLE_TYPE_K;
        case 'j':
        case 'J':
            return THERMOCOUPLE_TYPE_J;
        case 't':
        case 'T':
            return THERMOCOUPLE_TYPE_T;
        case 'n':
        case 'N':
            return THERMOCOUPLE_TYPE_N;
        case 's':
        case 'S':
            return THERMOCOUPLE_TYPE_S;
        case 'e':
        case 'E':
            return THERMOCOUPLE_TYPE_E;
        case 'b':
        case 'B':
            return THERMOCOUPLE_TYPE_B;
        case 'r':
        case 'R':
            return THERMOCOUPLE_TYPE_R;
    }

    return -1;
}

#endif
