/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#pragma once

#include <dracal/common/chrono.hpp>

#include <spdlog/details/log_msg.h>
#include <spdlog/sinks/base_sink.h>

#include <deque>
#include <mutex>
#include <string>
#include <vector>

namespace dracal::logging {

namespace utils {

/**
 * @brief Creates a comment section for the log file.
 *
 * Creates a comment section for the log file with the current timestamp, interval, and comment.
 *
 * @param now The timestamp to use.
 * @param interval The interval to use.
 * @param comment The comment to use.
 */
std::vector<std::string> create_comment_section(const chrono::time_point<chrono::system_clock> now,
                                                const chrono::milliseconds interval, const std::string &comment);

/**
 * @brief Creates a header section for the log file.
 *
 * Creates a header section for the log file with the separator, timestamp column names, product names, channel names,
 * channel descriptions, and unit symbols.
 *
 * @param separator The separator to use.
 * @param timestamp_column_names The timestamp column names to use.
 * @param product_names The product names to use.
 * @param channel_names The channel names to use.
 * @param channel_descriptions The channel descriptions to use.
 * @param unit_symbols The unit symbols to use.
 */
std::vector<std::string> create_header_section(const std::string &separator,
                                               const std::vector<std::string> &timestamp_column_names,
                                               const std::vector<std::string> &product_names,
                                               const std::vector<std::string> &channel_names,
                                               const std::vector<std::string> &channel_descriptions,
                                               const std::vector<std::string> &unit_symbols);

} // namespace utils

/**
 * @brief Custom memory sink for spdlog that stores log messages in a deque.
 *
 * This sink stores log messages in memory using a std::deque with a
 * maximum size limit. When the limit is reached, oldest messages are discarded.
 */
class memory_sink_mt : public spdlog::sinks::base_sink<std::mutex> {
  public:
    /**
     * @brief Constructor for memory sink.
     * @param max_lines Maximum number of lines to keep in memory (0 = no logging).
     */
    explicit memory_sink_mt(const size_t max_lines);

    /**
     * @brief Gets the maximum number of lines.
     *
     * @return Maximum number of lines (0 = no logging).
     */
    size_t max_lines() const { return _max_lines; }

    /**
     * @brief Gets a copy of all logged lines.
     *
     * Thread-safe access using the sink's internal mutex.
     *
     * @return Copy of the deque containing all logged lines.
     */
    std::deque<std::string> lines();

  protected:
    /**
     * @brief Sink implementation that formats and stores the log message.
     * @param msg The log message to be stored.
     */
    void sink_it_(const spdlog::details::log_msg &msg) override;

    /**
     * @brief Flush implementation (no-op for memory sink).
     */
    void flush_() override;

  private:
    const size_t _max_lines;
    std::deque<std::string> _lines;
};

} // namespace dracal::logging