/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#include <dracal/common/chrono.hpp>
#include <dracal/common/fmt.hpp>
#include <dracal/logging/utils.hpp>

namespace dracal::logging {

namespace utils {

std::vector<std::string> create_comment_section(const chrono::time_point<chrono::system_clock> now,
                                                const chrono::milliseconds interval, const std::string &comment) {

    const std::chrono::duration<double> seconds = interval;

    return {fmt::format("# Created: {}", common::format_timestamp_local_system_default(now)),
            fmt::format("# Interval: {:.6f} s", seconds.count()), fmt::format("# Comment: {}", comment), ""};
}

std::vector<std::string> create_header_section(const std::string &separator,
                                               const std::vector<std::string> &timestamp_column_names,
                                               const std::vector<std::string> &product_names,
                                               const std::vector<std::string> &channel_names,
                                               const std::vector<std::string> &channel_descriptions,
                                               const std::vector<std::string> &unit_symbols) {
    return {fmt::format("{}{}{}", fmt::join(timestamp_column_names, separator),
                        timestamp_column_names.size() == 0 ? "" : separator, fmt::join(product_names, separator)),
            fmt::format("{}{}", separator, fmt::join(channel_names, separator)),
            fmt::format("{}{}", separator, fmt::join(channel_descriptions, separator)),
            fmt::format("{}{}", separator, fmt::join(unit_symbols, separator))};
}

} // namespace utils

memory_sink_mt::memory_sink_mt(const size_t max_lines) : _max_lines(max_lines) {}

void memory_sink_mt::sink_it_(const spdlog::details::log_msg &msg) {
    if (_max_lines == 0) {
        return;
    }

    spdlog::memory_buf_t formatted;
    base_sink<std::mutex>::formatter_->format(msg, formatted);
    _lines.push_back(fmt::to_string(formatted));

    if (_lines.size() > _max_lines) {
        _lines.pop_front();
    }
}

void memory_sink_mt::flush_() {}

std::deque<std::string> memory_sink_mt::lines() {
    std::lock_guard<std::mutex> lock(mutex_);
    return _lines;
}

} // namespace dracal::logging