/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#pragma once

#include <dracal/common/platform.hpp>

#include <array>
#include <codecvt>
#include <cstdint>
#include <locale>
#include <span>
#include <string>
#include <vector>

namespace dracal::common {

/**
 * @brief Check if the string starts with the given prefix.
 * @param str The string to check.
 * @param prefix The prefix to look for.
 * @return True if starts with the prefix.
 */
bool starts_with(const std::string &str, const std::string &prefix) noexcept;

/**
 * @brief Check if the string contains with the given substring.
 * @param str The string to check.
 * @param substring The substring to look for.
 * @return True if contains with the substring.
 */
bool contains(const std::string &str, const std::string &substring) noexcept;

/**
 * @brief Check if the string contains at least one of the substrings.
 * @param str The string to check.
 * @param substrings The substrings to look for.
 * @return True if contains with the substring or if the substrings is empty.
 */
bool contains(const std::string &str, const std::vector<std::string> &substrings) noexcept;

/**
 * @brief Check if the string ends with the given postfix.
 * @param str The string to check.
 * @param postfix The postfix to look for.
 * @return True if ends with the postfix.
 */
bool ends_with(const std::string &str, const std::string &postfix) noexcept;

/**
 * @brief Convert the given string to lower case.
 * @param str The string to convert.
 * @return The lower case string.
 */
std::string to_lower(const std::string &str);

/**
 * @brief Convert the given string to lower case.
 * @param str The string to convert.
 * @param pos The position.
 * @return The lower case string.
 */
std::string to_lower(const std::string &str, const int pos);

/**
 * @brief Convert the given string to lower case at the given position.
 * @param str The string to convert.
 * @param pos The position.
 * @return The string with lower case character.
 */
std::string to_lower(const std::string &str, const unsigned int pos);

/**
 * @brief  Convert the given string to lower case at the given position.
 * @param str The string to convert.
 * @param pos The position.
 * @return The string with lower case character.
 */
std::string to_upper(const std::string &str);

/**
 * @brief Convert the given string to upper case at the given position.
 * @param str The string to convert.
 * @param pos The position.
 * @return The string with upper case character.
 */
std::string to_upper(const std::string &str, const int pos);

/**
 * @brief Convert the given string to upper case at the given position.
 * @param str The string to convert.
 * @param pos The position.
 * @return The string with upper case character.
 */
std::string to_upper(const std::string &str, const unsigned int pos);

/**
 * @brief Replace all the occurence of the FROM string to the TO string in the given string.
 * @param str The string to convert.
 * @param from The string to search for.
 * @param to The string to replace with.
 * @return The string with replacements.
 */
std::string replace(const std::string &str, const std::string &from, const std::string &to);

/**
 * @brief Trim left spaces in the given string. Works directly on the given string, no copy is made.
 * @param str The string to trim.
 */
void trim_left_inplace(std::string &str);

/**
 * @brief Trim right spaces in the given string. Works directly on the given string, no copy is made.
 * @param str The string to trim.
 */
void trim_right_inplace(std::string &str);

/**
 * @brief Trim both left and right spaces in the given string. Works directly on the given string, no copy is made.
 * @param str The string to trim.
 */
void trim_inplace(std::string &str);

/**
 * @brief Trim prefix in the given string. Works directly on the given string, no copy is made.
 * @param str The string to trim.
 * @param prefix The prefix to remove.
 */
void trim_prefix_inplace(std::string &str, const std::string &prefix);

/**
 * @brief Trim all before the prefix in the given string. Works directly on the given string, no copy is made.
 * @param str The string to trim.
 * @param prefix The prefix to remove.
 */
void trim_before_prefix_inplace(std::string &str, const std::string &prefix);

/**
 * @brief Trim postfix in the given string. Works directly on the given string, no copy is made.
 * @param str The string to trim.
 * @param postfix The postfix to remove.
 */
void trim_postfix_inplace(std::string &str, const std::string &postfix);

/**
 * @brief Removes any null terminators from the string. Works directly on the given string, no copy is made.
 *
 * Removes any null terminators ('\0') from the string `str`
 *
 * @param str The string from which to remove null terminators.
 */
void erase_null_terminators_inplace(std::string &str);

/**
 * @brief Trim left spaces in the given string. Creates a copy of the string, the orignial string is left untouched.
 * @param str The string to trim.
 * @return The trimmed string.
 */
std::string trim_left(const std::string &str);

/**
 * @brief Trim right spaces in the given string. Creates a copy of the string, the orignial string is left untouched.
 * @param str The string to trim.
 * @return The trimmed string.
 */
std::string trim_right(const std::string &str);

/**
 * @brief Trim both left and right spaces in the given string. Creates a copy of the string, the orignial string is left
 * untouched.
 * @param str The string to trim.
 * @return The trimmed string.
 */
std::string trim(const std::string &str);

/**
 * @brief Trim prefix in the given string. Creates a copy of the string, the orignial string is left untouched.
 * @param str The string to trim.
 * @param prefix The prefix to remove.
 * @return The trimmed string.
 */
std::string trim_prefix(const std::string &str, const std::string &prefix);

/**
 * @brief Trim all before the prefix in the given string. Creates a copy of the string, the orignial string is left
 * untouched.
 * @param str The string to trim.
 * @param prefix The prefix to remove.
 * @return The trimmed string.
 */
std::string trim_before_prefix(const std::string &str, const std::string &prefix);

/**
 * @brief Trim postfix in the given string. Creates a copy of the string, the original string is left untouched.
 * @param str The string to trim.
 * @param postfix The postfix to remove.
 * @return The trimmed string.
 */
std::string trim_postfix(const std::string &str, const std::string &postfix);

/**
 * @brief Trim the filename extension. Creates a copy of the string, the original string is left untouched.
 * @param filename The filename to trim.
 * @param keep_path Keep the path if present or not.
 * @return The trimmed string.
 */
std::string trim_extension(const std::string &filename, const bool keep_path = true);

/**
 * @brief Trim all characters after the first occurrence of the given delimiter.
 * @param str The string to trim.
 * @param delimiter The delimiter to search for.
 * @return The trimmed string or original string if delimiter not found.
 */
std::string trim_all_after_first(const std::string &str, const std::string &delimiter);

/**
 * @brief Trim all characters before the last occurrence of the given delimiter as well as the delimiter.
 * @param str The string to trim.
 * @param delimiter The delimiter to search for.
 * @return The trimmed string starting from last delimiter or original string if delimiter not found.
 */
std::string trim_all_before_last(const std::string &str, const std::string &delimiter);

/**
 * @brief Removes any null terminators from the string and returns the trimmed string. Creates a copy of the
 * string, the orignial string is left untouched.
 *
 * Removes any null terminators ('\0') from the string `str`.
 *
 * @param str The string from which to remove null terminators.
 * @return The trimmed string without null terminators.
 */
std::string erase_null_terminators(const std::string &str);

/**
 * @brief Prepends a prefix to the given string. Creates a copy of the string, the orignial string is left untouched.
 * @param str The string to be modified.
 * @param prefix The prefix to add at the start.
 * @param only_if_missing To avoid duplicated prefixes.
 * @return The modified string.
 */
std::string prepend_prefix(const std::string &str, const std::string &prefix, const bool only_if_missing = true);

/**
 * @brief Postpends a postfix to the given string. Creates a copy of the string, the orignial string is left untouched.
 * @param str The string to be modified.
 * @param postfix The postfix to add at the end.
 * @param only_if_missing To avoid duplicated postfixes.
 * @return The modified string.
 */
std::string append_postfix(const std::string &str, const std::string &postfix, const bool only_if_missing = true);

/**
 * @brief Splits the given string on the given delimeters.
 * @param str The string to split.
 * @param delims The delimeters to split on.
 * @param keep_empty_parts Keep the empty parts.
 * @return Vector containing the splited substrings.
 */
std::vector<std::string> split(const std::string &str, const std::string &delims = " ",
                               const bool keep_empty_parts = false);

/**
 * @brief Joins the given list with the provided delimeters.
 * @param list The string to join.
 * @param delims The delimeters to add in-between joined strings.
 * @return The joined string.
 */
std::string join(const std::vector<std::string> &list, const std::string &delims = " ");

/**
 * @brief Joins the given list with the provided delimeters from start index.
 * @param list The string to join.
 * @param delims The delimeters to add in-between joined strings.
 * @param start The start index to start join.
 * @return The joined string.
 */
std::string join_from(const std::vector<std::string> &list, const std::string &delims = " ", unsigned int start = 0);

#if defined(DRACAL_WINDOWS)
/**
 * @brief Converts std::wstring into std::string.
 * @param wstr The std::wstring to convert.
 * @return The converted std::string.
 */
std::string convert(const std::wstring &wstr);

/**
 * @brief Converts std::string into std::wstring.
 * @param str The std::string to convert.
 * @return The converted std::wstring.
 */
std::wstring convert(const std::string &str);
#endif

#if defined(DRACAL_HAS_SPAN)
std::string convect_utf16le_to_utf8(const std::span<const unsigned char> &data);
#endif

/**
 * @brief Extracts the executable name without the path and the extension if any.
 *
 * @param raw_executable_name The raw executable name with full path and extension. Typically argv[0].
 * @return The extracted executable name.
 */
std::string extract_executable_name(const char *raw_executable_name);

/**
 * @brief Generates a random UUID version 4 (RFC 4122).
 * @return A string containing a randomly generated UUID in the format "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".
 */
std::string uuid_v4();

} // namespace dracal::common