/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#pragma once

#include <dracal/common/platform.hpp>

#include <string>

#if defined(DRACAL_HAS_FILESYSTEM)
    #include <filesystem>
namespace fs = std::filesystem;
#elif defined(DRACAL_HAS_EXPERIMENTAL_FILESYSTEM)
    #include <experimental/filesystem>
namespace fs = std::experimental::filesystem;
#else
    // cppcheck-suppress preprocessorErrorDirective
    #error "No filesystem support available"
#endif

namespace dracal::common {

/**
 * @brief Reads the contents of a file into a string.
 *
 * Reads the contents of the file specified by `filename` into the string `text`.
 *
 * @param filename The name of the file to read.
 * @param text The string where the contents of the file will be stored.
 * @param quiet Quiet the logs on error.
 * @return `true` if the file is successfully read, `false` otherwise.
 */
bool read_file(const std::string &filename, std::string &text, const bool quiet = false) noexcept;

/**
 * @brief Reads the contents of a file into a string.
 *
 * Reads the contents of the file specified by `filename` into the string `text`.
 *
 * @param file The file to read.
 * @param text The string where the contents of the file will be stored.
 * @param quiet Quiet the logs on error.
 * @return `true` if the file is successfully read, `false` otherwise.
 */
bool read_file(const fs::path &file, std::string &text, const bool quiet = false) noexcept;

/**
 * @brief Writes a string to a file.
 *
 * Writes the contents of the string `text` to the file specified by `filename`.
 *
 * @param filename The name of the file to write to.
 * @param text The string containing the data to write to the file.
 * @param quiet Quiet the logs on error.
 * @return `true` if the file is successfully written, `false` otherwise.
 */
bool write_file(const std::string &filename, const std::string &text, const bool quiet = false) noexcept;

/**
 * @brief Writes a string to a file.
 *
 * Writes the contents of the string `text` to the file specified by `filename`.
 *
 * @param file The file to write to.
 * @param text The string containing the data to write to the file.
 * @param quiet Quiet the logs on error.
 * @return `true` if the file is successfully written, `false` otherwise.
 */
bool write_file(const fs::path &file, const std::string &text, const bool quiet = false) noexcept;

} // namespace dracal::common