/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#include <dracal/common/fs.hpp>
#include <dracal/common/log.hpp>

#include <fstream>

namespace dracal::common {

bool read_file(const std::string &filename, std::string &text, const bool quiet) noexcept {
    try {
        const fs::path complete_filename = filename;

        if (!fs::exists(complete_filename)) {
            if (!quiet) {
                dracal_warn("unable to read file({}) with complete path({}): file doesn't exists", filename,
                            complete_filename.string());
            }

            return false;
        }

        std::ifstream file(complete_filename, std::ios::in | std::ios::binary);
        file.exceptions(std::ifstream::failbit);
        const auto size = fs::file_size(complete_filename);
        text = std::string(static_cast<size_t>(size), '\0');
        file.read(text.data(), static_cast<std::streamsize>(size));
    } catch (const std::ios_base::failure &fail) {
        if (!quiet) {
            dracal_warn("failed to read file({}) with error: ({})", filename, fail.what());
        }

        return false;
    } catch (...) {
        if (!quiet) {
            dracal_warn("an unexpected exception was thrown");
        }

        return false;
    }

    return true;
}

bool read_file(const fs::path &file, std::string &text, const bool quiet) noexcept {
    try {
        if (!fs::exists(file)) {
            if (!quiet) {
                dracal_warn("unable to read file({}): file doesn't exists", file.string());
            }

            return false;
        }

        std::ifstream f(file, std::ios::in | std::ios::binary);
        f.exceptions(std::ifstream::failbit);
        const auto size = fs::file_size(file);
        text = std::string(static_cast<size_t>(size), '\0');
        f.read(text.data(), static_cast<std::streamsize>(size));
    } catch (const std::ios_base::failure &fail) {
        if (!quiet) {
            dracal_warn("failed to read file({}) with error: ({})", file.string(), fail.what());
        }

        return false;
    } catch (...) {
        if (!quiet) {
            dracal_warn("an unexpected exception was thrown");
        }

        return false;
    }

    return true;
}

bool write_file(const std::string &filename, const std::string &text, const bool quiet) noexcept {
    try {
        std::ofstream file(fs::path(filename), std::ios::out);
        file.exceptions(std::ofstream::failbit);
        file.write(text.c_str(), static_cast<std::streamsize>(text.size()));
    } catch (const std::ios_base::failure &fail) {
        if (!quiet) {
            dracal_warn("failed to write to file({}) with error: ({})", filename, fail.what());
        }

        return false;
    } catch (...) {
        if (!quiet) {
            dracal_warn("an unexpected exception was thrown");
        }

        return false;
    }

    return true;
}

bool write_file(const fs::path &file, const std::string &text, const bool quiet) noexcept {
    try {
        std::ofstream f(file, std::ios::out);
        f.exceptions(std::ofstream::failbit);
        f.write(text.c_str(), static_cast<std::streamsize>(text.size()));
    } catch (const std::ios_base::failure &fail) {
        if (!quiet) {
            dracal_warn("failed to write to file({}) with error: ({})", file.string(), fail.what());
        }

        return false;
    } catch (...) {
        if (!quiet) {
            dracal_warn("an unexpected exception was thrown");
        }

        return false;
    }

    return true;
}

} // namespace dracal::common