/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#pragma once

#include <array>
#include <chrono>
#include <string>

namespace chrono = std::chrono;

namespace dracal::common {

/**
 * Format a timestamp in local time with the format YYYY-MM-DDTHH:MM:SS
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::string format_timestamp_local_YYYYMMDDTHHMMSS(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format YYYY-MM-DDTHH:MM:SS.mmm
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::string format_timestamp_local_YYYYMMDDTHHMMSSmmm(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format YYYY-MM-DD and HH:MM:SS
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */

std::array<std::string, 2> format_timestamp_local_YYYYMMDD_HHMMSS(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format YYYY-MM-DD and HH:MM:SS.mmm
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::array<std::string, 2> format_timestamp_local_YYYYMMDD_HHMMSSmmm(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format HH:MM:SS
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::string format_timestamp_local_HHMMSS(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format HH:MM:SS.mmm
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::string format_timestamp_local_HHMMSSmmm(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format milliseconds from epoch
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::string format_timestamp_milliseconds_from_epoch(const chrono::time_point<chrono::system_clock> tp);

/**
 * Format a timestamp in local time with the format system default
 *
 * @param tp The timestamp to format.
 * @return The formatted timestamp.
 */
std::string format_timestamp_local_system_default(const chrono::time_point<chrono::system_clock> tp);

} // namespace dracal::common
