/* -*- mode: c++ -*-
 * Copyright 2025 Dracal Technologies Inc. All rights reserved.
 */

#include <dracal/common/chrono.hpp>
#include <dracal/common/fmt.hpp>
#include <dracal/common/platform.hpp>

#if defined(DRACAL_APPLE)
    #include <ctime>
namespace {
std::tm to_local_time(const chrono::time_point<chrono::system_clock> tp) {
    const auto time_t_val = chrono::system_clock::to_time_t(tp);
    std::tm local_tm;
    localtime_r(&time_t_val, &local_tm);
    return local_tm;
}
} // anonymous namespace
#endif

#if defined(DRACAL_LINUX)
    #include <ctime>
namespace {
std::tm to_local_time(const chrono::time_point<chrono::system_clock> tp) {
    const auto time_t_val = chrono::system_clock::to_time_t(tp);
    std::tm local_tm;
    localtime_r(&time_t_val, &local_tm);
    return local_tm;
}
} // anonymous namespace
#endif

namespace dracal::common {

std::string format_timestamp_local_YYYYMMDDTHHMMSS(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    return fmt::format("{:%Y-%m-%dT%H:%M:%S}", local_tm);
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    return fmt::format("{:%Y-%m-%dT%H:%M:%S}", local_tm);
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return fmt::format("{:%Y-%m-%dT%H:%M:%S}", local_time);
#endif
}

std::string format_timestamp_local_YYYYMMDDTHHMMSSmmm(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    return fmt::format("{:%Y-%m-%dT%H:%M:%S}.{:03}", local_tm, millis.count());
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    return fmt::format("{:%Y-%m-%dT%H:%M:%S}.{:03}", local_tm, millis.count());
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return fmt::format("{:%Y-%m-%dT%H:%M:%S}.{:03}", local_time, millis.count());
#endif
}

std::array<std::string, 2> format_timestamp_local_YYYYMMDD_HHMMSS(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    return {fmt::format("{:%Y-%m-%d}", local_tm), fmt::format("{:%H:%M:%S}", local_tm)};
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    return {fmt::format("{:%Y-%m-%d}", local_tm), fmt::format("{:%H:%M:%S}", local_tm)};
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return {fmt::format("{:%Y-%m-%d}", local_time), fmt::format("{:%H:%M:%S}", local_time)};
#endif
}

std::array<std::string, 2>
format_timestamp_local_YYYYMMDD_HHMMSSmmm(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    return {fmt::format("{:%Y-%m-%d}", local_tm), fmt::format("{:%H:%M:%S}.{:03}", local_tm, millis.count())};
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    return {fmt::format("{:%Y-%m-%d}", local_tm), fmt::format("{:%H:%M:%S}.{:03}", local_tm, millis.count())};
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return {fmt::format("{:%Y-%m-%d}", local_time), fmt::format("{:%H:%M:%S}.{:03}", local_time, millis.count())};
#endif
}

std::string format_timestamp_local_HHMMSS(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    return fmt::format("{:%H:%M:%S}", local_tm);
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    return fmt::format("{:%H:%M:%S}", local_tm);
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return fmt::format("{:%H:%M:%S}", local_time);
#endif
}

std::string format_timestamp_local_HHMMSSmmm(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    return fmt::format("{:%H:%M:%S}.{:03}", local_tm, millis.count());
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    return fmt::format("{:%H:%M:%S}.{:03}", local_tm, millis.count());
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch()) % 1000;
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return fmt::format("{:%H:%M:%S}.{:03}", local_time, millis.count());
#endif
}

std::string format_timestamp_milliseconds_from_epoch(const chrono::time_point<chrono::system_clock> tp) {
    const auto millis = chrono::duration_cast<chrono::milliseconds>(tp.time_since_epoch());
    return fmt::format("{}", millis.count());
}

std::string format_timestamp_local_system_default(const chrono::time_point<chrono::system_clock> tp) {
#if defined(DRACAL_APPLE)
    const auto local_tm = to_local_time(tp);
    return fmt::format("{:%c}", local_tm);
#elif defined(DRACAL_LINUX)
    const auto local_tm = to_local_time(tp);
    return fmt::format("{:%c}", local_tm);
#else
    const auto seconds_tp = chrono::time_point_cast<chrono::seconds>(tp);
    const auto zoned = chrono::zoned_time{chrono::current_zone(), seconds_tp};
    const auto local_time = zoned.get_local_time();
    return fmt::format("{:%c}", local_time);
#endif
}

} // namespace dracal::common