#include <QDebug>
#include <QEventLoop>
#include <QMutexLocker>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QUrl>

#include "Config.h"
#include "OriginTimestamp.h"
#include "TDeviceManager.h"
#include "TUSBProvider.h"
#include "USBThread.h"
#include "chip.h"
#include "usbtenki.h"
#include "usbtenki_provider.h"

static const Version TICKER_VERSION = {.major = 0, .minor = 0};

static const Quantity TICKER_QUANTITY = {UNIT_UNKNOWN, QUANTITY_TYPE_ERROR, CHIP_ERROR_TICKER_DEVICE};

TUSBProvider::TUSBProvider() : TProvider("USB Devices") {

    USBThread::instance().setPollInterval(pollInterval);
    USBThread::instance().setVirtualOptions(virtualOptions.flags, virtualOptions.standard_sea_level_pressure);

    tickerDevice = new TDevice(this, TDevice::TICKER_SERIAL_NUMBER, TDevice::TICKER_SERIAL_NUMBER, TICKER_VERSION);

    tickerDevice->timestamp = OriginTimestamp::instance().value();
    tickerDevice->flags = (DEVICE_FLAG_ALIVE | DEVICE_FLAG_TICKER);

    tickerDevice->channels.resize(1);
    tickerDevice->channels[0] = new TChannel(tickerDevice, 0, USBTENKI_CHIP_TICKER, TDevice::TICKER_CHANNEL_ID);
    tickerDevice->channels[0]->quantity = TICKER_QUANTITY;
    tickerDevice->channels[0]->calibratedQuantity = TICKER_QUANTITY;

    TDeviceManager::instance().addDevice(tickerDevice);
    TDeviceManager::instance().connectDevice(tickerDevice);
}

TUSBProvider::~TUSBProvider() {
    // Singleton lives forever
}

int TUSBProvider::start() {

    USBThread &thread = USBThread::instance();

    if (!thread.isRunning()) {
        thread.start();
    }

    return 0;
}

void TUSBProvider::shutdown() { USBThread::instance().shutdown(); }

int TUSBProvider::getDeviceCount() {

    QMutexLocker locker(&mutex);

    return devicesBySerialNumber.size();
}

QVector<TDevice *> TUSBProvider::getDevices() {

    QMutexLocker locker(&mutex);

    QVector<TDevice *> results(devicesBySerialNumber.size());
    int i = 0;

    QMapIterator<QString, TDevice *> it(devicesBySerialNumber);
    while (it.hasNext()) {
        it.next();
        results[i++] = it.value();
    }

    return results;
}

TDevice *TUSBProvider::findDevice(const QString &serial) {

    QMutexLocker locker(&mutex);

    return devicesBySerialNumber.value(serial);
}

int TUSBProvider::set_DXC120_ASC(const QString &serialNumber, int value) {

    QMutexLocker locker(&(USBThread::instance()._mutex));

    QByteArray serialNumberByteArray = serialNumber.toLocal8Bit();

    USBTenki_dev_handle handle = usbtenki_provider_find_handle(serialNumberByteArray.data());

    if (!handle) {
        return -1;
    }

    int res = usbtenki_set_dxc120_asc(handle, value);

    usbtenki_closeDevice(handle);
    return res;
}

int TUSBProvider::get_DXC120_ASC(const QString &serialNumber) {

    QMutexLocker locker(&(USBThread::instance()._mutex));

    QByteArray serialNumberByteArray = serialNumber.toLocal8Bit();

    USBTenki_dev_handle handle = usbtenki_provider_find_handle(serialNumberByteArray.data());

    if (!handle) {
        return -1;
    }

    int res = usbtenki_get_dxc120_asc(handle);

    usbtenki_closeDevice(handle);
    return res;
}

int TUSBProvider::set_DXC120_FRC(const QString &serialNumber, int value) {

    QMutexLocker locker(&(USBThread::instance()._mutex));

    QByteArray serialNumberByteArray = serialNumber.toLocal8Bit();

    USBTenki_dev_handle handle = usbtenki_provider_find_handle(serialNumberByteArray.data());

    if (!handle) {
        return -1;
    }

    int res = usbtenki_set_dxc120_frc(handle, value);

    usbtenki_closeDevice(handle);
    return res;
}

int TUSBProvider::set_DXC220_KZ(const QString &serialNumber, int value) {

    QMutexLocker locker(&(USBThread::instance()._mutex));

    QByteArray serialNumberByteArray = serialNumber.toLocal8Bit();

    USBTenki_dev_handle handle = usbtenki_provider_find_handle(serialNumberByteArray.data());

    if (!handle) {
        return -1;
    }

    int res = usbtenki_set_dxc220_kz(handle, value);

    usbtenki_closeDevice(handle);
    return res >= 0;
}

bool TUSBProvider::isServiceRunning() {
    static const bool running = []() -> bool {
        return false;
        // QNetworkAccessManager manager;
        // const QString port = loadDracalUsbServicePort();
        // const QString url = QString("http://localhost:%1/dracal-service-info").arg(port);
        // QNetworkRequest request{QUrl(url)};

        // QNetworkReply *reply = manager.get(request);

        // // Block until the request is finished
        // QEventLoop loop;
        // QObject::connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
        // loop.exec();

        // bool result = (reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() == 200);

        // reply->deleteLater();
        // return result;
    }();

    return running;
}

void TUSBProvider::setPollInterval(int ms) { USBThread::instance().setPollInterval(ms); }

void TUSBProvider::setVirtualOptions(unsigned int flags, double slp) {

    USBThread::instance().setVirtualOptions(flags, slp);
}

void TUSBProvider::setUserCalibration(const QString &serialNumber, int channelIndex, bool on) {

    USBThread::instance().setUserCalibration(serialNumber, channelIndex, on);
}

int TUSBProvider::setUserCalibrationPoint(TChannel *channel, int id, double x, double y) {
    QMutexLocker locker(&(USBThread::instance()._mutex));

    TDevice *device = channel->device;

    QByteArray serialNumberByteArray = device->serialNumber.toLocal8Bit();

    USBTenki_dev_handle handle = usbtenki_provider_find_handle(serialNumberByteArray.data());

    if (!handle) {
        return -1;
    }

    Point point = {x, y};
    int res = usbtenki_setUserCalibrationPoint(handle, channel->index, id, &point);

    usbtenki_closeDevice(handle);
    return res;
}

void TUSBProvider::tick(int64_t timestamp) {

    tickerDevice->timestamp = timestamp;

    TDeviceManager::instance().updateDevice(tickerDevice);
}

void TUSBProvider::addSource(Source *source) {

    QString serialNumber;
    TDevice *device;

    {
        QMutexLocker locker(&mutex);

        serialNumber = source->device->serial_number;
        device = devicesBySerialNumber.value(serialNumber);

        if (device) {
            return;
        }

        device = new TDevice(this, source);
        devicesBySerialNumber.insert(serialNumber, device);
    }

    TDeviceManager::instance().addDevice(device);

    if (device->isAlive()) {
        TDeviceManager::instance().connectDevice(device);
    } else {
        disconnectedSerialNumbers.insert(serialNumber);
    }
}

void TUSBProvider::updateSource(Source *source) {

    TDevice *device;
    QString serialNumber;

    {
        QMutexLocker locker(&mutex);

        serialNumber = source->device->serial_number;
        device = devicesBySerialNumber.value(serialNumber);

        if (!device) {
            return; // should never happen
        }

        device->update(source);
    }

    if (device->isAlive()) {
        if (disconnectedSerialNumbers.remove(serialNumber)) {
            TDeviceManager::instance().connectDevice(device);
        }
        TDeviceManager::instance().updateDevice(device);
    } else if (!disconnectedSerialNumbers.contains(serialNumber)) {
        disconnectedSerialNumbers.insert(serialNumber);
        TDeviceManager::instance().disconnectDevice(device);
    }
}
