#include <QSettings>

#include "ConfigComboBox.h"
#include "ConfigPanel.h"

ConfigComboBox::ConfigComboBox(const QString &key, const QStringList &items, int defaultIndex) : key(key) {
    addItems(items);

    QSettings settings;
    int index = settings.value(key, defaultIndex).toInt();

    setCurrentIndex(index);
    connect(this, SIGNAL(currentIndexChanged(int)), this, SLOT(onChange(int)));
}

ConfigComboBox::~ConfigComboBox() {}

void ConfigComboBox::onChange(int index) {

    QSettings settings;
    settings.setValue(key, index);
}

//
// Field separator
//

const char *FieldSeparatorComboBox::values[] = {
    ", ",
    "\t",
    " ",
    "; ",
};

FieldSeparatorComboBox::FieldSeparatorComboBox()
    : ConfigComboBox("logger/format", {
                                          "Comma",
                                          "Tab",
                                          "Space",
                                          "Semicolon",
                                      }) {
    onChange(currentIndex());
}

void FieldSeparatorComboBox::onChange(int index) {

    ConfigComboBox::onChange(index);

    ConfigPanel::instance().getCSVOptions().separator_str = values[index];
}

//
// Decimal point
//

DecimalPointComboBox::DecimalPointComboBox()
    : ConfigComboBox("logger/decimal_point", {"System Default", "Period: .", "Comma: ,"}) {
    onChange(currentIndex());
}

void DecimalPointComboBox::onChange(int index) {

    ConfigComboBox::onChange(index);

    if (index == 2) { // comma
        ConfigPanel::instance().getCSVOptions().flags |= CSV_FLAG_DECIMAL_COMMA;
    } else {
        ConfigPanel::instance().getCSVOptions().flags &= ~CSV_FLAG_DECIMAL_COMMA;
    }
}

//
// Time format
//

const int8_t TimeFormatComboBox::values[] = {
    CSV_TIME_NONE,
    CSV_TIME_SYSTEM_DEFAULT,
    CSV_TIME_SYSTEM_DEFAULT,
    CSV_TIME_ISO_8601_LONG,
    CSV_TIME_ISO_8601_LONG_DUAL,
    CSV_TIME_ISO_8601_SHORT,
    CSV_TIME_ISO_8601_LONG_MS,
    CSV_TIME_ISO_8601_LONG_MS_DUAL,
    CSV_TIME_ISO_8601_SHORT_MS,
    CSV_TIME_EPOCH_MS,
};

TimeFormatComboBox::TimeFormatComboBox()
    : ConfigComboBox("logger/timestamp",
                     {
                         "None",
                         "Short system default",
                         "Long system default",
                         "ISO 8601 Date and time in one field: YYYY-mm-ddTHH:MM:SS",
                         "ISO 8601 Date and time in two fields: YYYY-mm-dd ; HH:MM:SS",
                         "ISO 8601 Time only: HH:MM:SS",
                         "ISO 8601 Date and time in one field: YYYY-mm-ddTHH:MM:SS.SSS",
                         "ISO 8601 Date and time in two fields: YYYY-mm-dd ; HH:MM:SS.SSS",
                         "ISO 8601 Time only: HH:MM:SS.SSS",
                         "Milliseconds since Epoch",
                     },
                     6) {
    onChange(currentIndex());
}

void TimeFormatComboBox::onChange(int index) {

    ConfigComboBox::onChange(index);

    ConfigPanel::instance().getCSVOptions().time_format = values[index];
}

//
// Error string
//

const char *ErrorStringComboBox::values[] = {
    "", nullptr, "0", "-1", "error",
};

ErrorStringComboBox::ErrorStringComboBox()
    : ConfigComboBox("logger/on_error", {
                                            "Write empty field",
                                            "Repeat previous value",
                                            "Write 0 in field",
                                            "Write -1 in field,",
                                            "Write \"error\" in field",
                                        }) {
    onChange(currentIndex());
}

void ErrorStringComboBox::onChange(int index) {

    ConfigComboBox::onChange(index);

    ConfigPanel::instance().getCSVOptions().error_str = values[index];
}
