#include <QDebug>
#include <QDoubleValidator>
#include <QEventLoop>
#include <QGridLayout>
#include <QJsonDocument>
#include <QJsonObject>
#include <QMessageBox>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

#include "CalibrationEditDialog_DXC120.h"
#include "Config.h"
#include "TUSBProvider.h"

#define FRESH_AIR_VALUE_MIN 400
#define FRESH_AIR_VALUE_DEFAULT 450
#define FRESH_AIR_VALUE_MAX 2000

CalibrationEditDialog_DXC120::CalibrationEditDialog_DXC120(QWidget *parent, const QString &serialNumber,
                                                           const bool localTenkinet)
    : QDialog(parent), serialNumber(serialNumber), localTenkinet(localTenkinet) {
    QDoubleValidator *valueValidator;

    btn_close = new QPushButton(tr("Close"));

    connect(btn_close, SIGNAL(clicked()), this, SLOT(close()));
    setWindowTitle(tr("Calibrate channel"));

    QGridLayout *gridLayout = new QGridLayout();

    QLabel *freshAirLabel = new QLabel("Fresh air (ppm)");
    freshAirEdit = new QLineEdit();
    valueValidator = new QDoubleValidator(this);
    valueValidator->setRange(FRESH_AIR_VALUE_MIN, FRESH_AIR_VALUE_MAX, 0);
    freshAirEdit->setValidator(valueValidator);
    btn_set_fresh_air = new QPushButton("Set fresh air");
    connect(btn_set_fresh_air, &QPushButton::clicked, this, &CalibrationEditDialog_DXC120::setFreshAir);

    gridLayout->addWidget(freshAirLabel, 0, 0);
    gridLayout->addWidget(freshAirEdit, 0, 1);
    gridLayout->addWidget(btn_set_fresh_air, 0, 2);

    cb_disable = new QCheckBox("Disable auto-calibration");

    connect(cb_disable, &QCheckBox::stateChanged, this, &CalibrationEditDialog_DXC120::autoCalibrationStateChanged);

    QHBoxLayout *lay_btns = new QHBoxLayout();
    lay_btns->addStretch();
    lay_btns->addWidget(btn_close);

    QVBoxLayout *lay_vert = new QVBoxLayout();
    lay_vert->addSpacing(10);
    lay_vert->addWidget(cb_disable);
    lay_vert->addSpacing(10);
    lay_vert->addStretch();
    lay_vert->addLayout(gridLayout);
    lay_vert->addLayout(lay_btns);

    setLayout(lay_vert);
    setModal(true);

    refresh();
}

CalibrationEditDialog_DXC120::~CalibrationEditDialog_DXC120() {}

void CalibrationEditDialog_DXC120::refresh() {

    freshAirEdit->setText(QString::number(FRESH_AIR_VALUE_DEFAULT));

    int ascEnabled = -1;

    ascEnabled = TUSBProvider::instance().get_DXC120_ASC(serialNumber);

    if (ascEnabled < 0) {
        // Could not get ASC from sensor...
        cb_disable->setEnabled(false);
        freshAirEdit->setEnabled(false);
        btn_set_fresh_air->setEnabled(false);
    } else {
        cb_disable->setChecked(!ascEnabled);
        cb_disable->setEnabled(true);
        freshAirEdit->setEnabled(true);
        btn_set_fresh_air->setEnabled(true);
    }
}

void CalibrationEditDialog_DXC120::autoCalibrationStateChanged(int state) {
    // Checked:   auto-calibration disabled
    // Unchecked: auto-calibration enabled

    int enabled = (state == Qt::Unchecked) ? 1 : 0;

    TUSBProvider::instance().set_DXC120_ASC(serialNumber, enabled);
}

void CalibrationEditDialog_DXC120::setFreshAir() {
    if (freshAirEdit->hasAcceptableInput()) {
        QString value = freshAirEdit->text();
        TUSBProvider::instance().set_DXC120_FRC(serialNumber, value.toInt());
    } else {
        QMessageBox msgBox;
        QString message;
        message.sprintf("Value must be between %d and %d ppm", FRESH_AIR_VALUE_MIN, FRESH_AIR_VALUE_MAX);
        msgBox.setIcon(QMessageBox::Information);
        msgBox.setText(message);
        msgBox.exec();
    }
}

void CalibrationEditDialog_DXC120::showEvent(QShowEvent *event) {

    refresh();
    QDialog::showEvent(event);
}
