#include <QMutexLocker>
#include <QDebug>

#include "USBThread.h"
#include "DaemonThreadManager.h"
#include "TUSBProvider.h"
#include "OriginTimestamp.h"
#include "timestamp.h"
#include "usbtenki_provider.h"

#define POLL_DELAY_DEFAULT 4000
#define POLL_DELAY_MAX 20000


USBThread::USBThread() :
    interval(0),
    pollDelay(POLL_DELAY_DEFAULT)
{
    DaemonThreadManager::instance().addThread(this);
}

USBThread::~USBThread() {

}

void USBThread::run() {

    TUSBProvider& provider = TUSBProvider::instance();
    int64_t origin = OriginTimestamp::instance().value();
    int64_t target = origin;

    if (!provider.isServiceRunning()) {
        // Initialize USBTenki library
        {
            QMutexLocker locker(&mutex);

            if (usbtenki_provider_init()) {
                return;
            }

            usbtenki_provider_poll();

            // Add connected devices
            const int n = usbtenki_provider_get_source_count();

            now = timestamp_now();

            for (int i = 0; i < n; i++) {
                Source *source = usbtenki_provider_get_source(i);
                source->timestamp = now;
                sourcesBySerialNumber.insert(source->device->serial_number, source);
                provider.addSource(source);
            }
        }
    }

    poll(now);

    // Main loop until shutdown() is called
    while (true) {

        now = timestamp_now();

        {
            QMutexLocker locker(&mutex);
            if (state == SHUTDOWN) {
                break;
            }

            if (requestedInterval != interval) {
                interval = requestedInterval;
                if (interval > 0) {
                    target = nearestTarget(origin, interval);
                }
            }
        }

        unsigned long sleepDelay = 1000000;  // 1 second

        if (interval == 0) {
            usleep(sleepDelay);
            continue;
        }

        int64_t delay = pollDelay;

        // Compute time remaining (delta) until target, within delay margin
        int64_t delta = target - now - (delay / 2);

        // If we've overshot the target by too much, jump to the nearest target
        if (delta + POLL_DELAY_MAX < 0) {
            target = nearestTarget(target, interval);
            delta = target - now - (delay / 2);
        }

        // If there is time remaining, don't get data yet. Just sleep until the next loop.
        if (delta > 0) {
            if (delta < sleepDelay) {
                sleepDelay = delta;
            }
            usleep(sleepDelay);
            continue;
        }

        // Get data
        poll(target);

        // Next target
        target += interval;

    }

    if (!provider.isServiceRunning()) {
        // Clean up and terminate thread
        mutex.lock();
        usbtenki_provider_shutdown();
        mutex.unlock();
    }
}

int64_t USBThread::nearestTarget(int64_t origin, uint32_t interval) {

    int64_t delta = origin - now;
    int64_t offset = delta % interval;

    if (offset < 0) {
        offset += interval;
    }

    return now + offset;

}

void USBThread::poll(int64_t timestamp) {

    TUSBProvider& provider = TUSBProvider::instance();

    if (!provider.isServiceRunning()) {
        mutex.lock();
        usbtenki_provider_poll();
        mutex.unlock();
    }

    // How long did it take us to poll devices?
    // A negative value means the system clock was set back during polling.
    // The clock can also have been set forward, resulting in a large value.

    int64_t before = now;
    int64_t after = timestamp_now();
    int64_t delay = after - before;

    if (delay > 0) {

        if (delay > POLL_DELAY_MAX) {
            delay = POLL_DELAY_MAX;
        }

        // Estimate the average poll delay
        // Exponential Moving Average with alpha=0.5

        uint64_t avg_old = pollDelay;
        uint64_t avg_new = ((uint32_t)delay / 2) + (avg_old / 2);
        pollDelay = avg_new;

        #ifdef TRACE
        double a = avg_old;
        double b = delay;
        double error = ((b - a) / a) * 100.0;
        if (error < 0.0) { error = -error; }
        QString msg;
        msg.sprintf(
            "Poll delay: old = %llu, cur = %lld, new = %llu [error = %.4f %%]",
            avg_old,
            delay,
            avg_new,
            error
        );
        qDebug() << msg;
        #endif

    }
    #ifdef TRACE
    else {
        qDebug() << "Poll delay was negative";
    }
    #endif

    // Update provider
    provider.tick(timestamp);

    if (!provider.isServiceRunning()) {
        mutex.lock();

        const int n = usbtenki_provider_get_source_count();

        for (int i = 0; i < n; i++) {
            Source *source = usbtenki_provider_get_source(i);
            source->timestamp = timestamp;
            provider.updateSource(source);
        }

        mutex.unlock();
    }

}

void USBThread::shutdown() {

    mutex.lock();
    state = SHUTDOWN;
    mutex.unlock();

}

void USBThread::setPollInterval(unsigned int ms) {

    mutex.lock();
    requestedInterval = 1000 * ms;
    mutex.unlock();

}

void USBThread::setVirtualOptions(unsigned int flags, double slp) {

    mutex.lock();
    usbtenki_virtual_options.flags = flags;
    usbtenki_virtual_options.standard_sea_level_pressure = slp;
    mutex.unlock();

}

void USBThread::setUserCalibration(const QString& serialNumber, int channelIndex, bool on) {

    QMutexLocker locker(&mutex);

    Source *source = sourcesBySerialNumber.value(serialNumber);
    if (!source) {
        return;
    }

    source_set_user_calibration(source, channelIndex, on);

}
