#include <stdio.h>
#include <string.h>

#include <QMutexLocker>
#include <QDebug>

#include "LoggingThread.h"
#include "DaemonThreadManager.h"
#include "TDeviceManager.h"
#include "ConfigPanel.h"
#include "OriginTimestamp.h"
#include "MathProvider.h"
#include "usbtenki_version.h"
#include "chip.h"


static const char FILE_MODE_WRITE[] = "w";
static const char FILE_MODE_APPEND[] = "a";


static char* new_char_array(const QString& s) {

    QByteArray bytes = s.toUtf8();
    unsigned int len = qstrlen(bytes);
    char *result = new char[len + 1];
    memcpy(result, bytes.constData(), len);
    result[len] = '\0';

    return result;

}


LoggingThread::LoggingThread(
    const QVector<QString>& channelIDs,
    const QString& path,
    const QString& comment,
    int interval,
    bool append
) :
    path(path),
    channelCount(channelIDs.size()),
    csvopt(ConfigPanel::instance().getCSVOptions()),
    fileMode(append ? FILE_MODE_APPEND : FILE_MODE_WRITE),
    origin(OriginTimestamp::instance().value()),
    interval_us(interval * 1000),  // ms -> us
    lineCounter(0)
{

    TDeviceManager& dm = TDeviceManager::instance();

    csvopt.file = nullptr;
    csvopt.datalog = &datalog;

    // Add ticker device
    QVector<QString> channelIDsWithTicker(channelIDs);
    channelIDsWithTicker.append(TDevice::TICKER_CHANNEL_ID);
    TDeviceSignal *tickerSignal = dm.getDeviceSignal(TDevice::TICKER_SERIAL_NUMBER);
    connect(tickerSignal, &TDeviceSignal::deviceUpdated, this, &LoggingThread::onDeviceUpdated);

    // Initialize channel headers
    list_init(&channelHeaders);
    list_grow(&channelHeaders, channelIDs.size());

    // Create DataTable
    table = new DataTable(channelIDsWithTicker, false);
    table->connect(0, TDevice::TICKER_CHANNEL_ID);

    datalog.interval = interval_us;
    datalog.comment = new_char_array(comment);

    list_init(&datalog.devices);
    list_init(&datalog.channels);
    list_grow(&datalog.channels, channelIDs.size());

    for (int i = 0; i < channelIDs.size(); i++) {

        QString id = channelIDs[i];

        TChannel *inputChannel = dm.getChannel(id);
        if (!inputChannel) {
            continue;
        }

        TDevice *inputDevice = inputChannel->device;
        if (inputDevice->isAlive()) {
            table->connect(inputDevice->timestamp, i);
            table->insert(inputDevice->timestamp, i, inputChannel->calibratedQuantity);
        }
        
        TDeviceSignal *signal = deviceSignalBySerial.value(inputDevice->serialNumber);
        if (!signal) {
            signal = dm.getDeviceSignal(inputDevice->serialNumber);
            if (!signal) {
                continue;
            }
            deviceSignalBySerial.insert(inputDevice->serialNumber, signal);
            connect(signal, &TDeviceSignal::deviceConnected, this, &LoggingThread::onDeviceConnected);
            connect(signal, &TDeviceSignal::deviceDisconnected, this, &LoggingThread::onDeviceDisconnected);
            connect(signal, &TDeviceSignal::deviceUpdated, this, &LoggingThread::onDeviceUpdated);
        }

        Channel *channel = new Channel();
        channel->chip_id = inputChannel->chip_id;
        channel->quantity = inputChannel->calibratedQuantity;

        list_add(&datalog.channels, channel);

        // Custom channel header

        CSV_Channel_Header *ch = new CSV_Channel_Header();
        ch->name = new_char_array(inputDevice->productName);
        ch->id = new_char_array(inputChannel->id);

        ch->description = new_char_array(dm.getAlias(inputChannel->id));

        if (inputDevice->isMath()) {
            ch->unit = MathProvider::instance().getUnit(inputChannel->index);
        }
        else {
            unit_t nativeUnit = inputChannel->getNativeUnit();
            unit_t prefUnit = csvopt.units[unit_category(nativeUnit)];
            if (prefUnit == UNIT_SENSOR_DEFAULT) {
                prefUnit = nativeUnit;
            }
            ch->unit = prefUnit;
        }

        list_add(&channelHeaders, ch);

    }

    // Initialize event queue
    queue_init(&events);
    queue_grow(&events, 64);

    // Disable unit conversion in csv module since input quantities already have the correct unit
    memset(csvopt.units, UNIT_SENSOR_DEFAULT, sizeof(csvopt.units));

    DaemonThreadManager::instance().addThread(this);

}

LoggingThread::~LoggingThread() {

    DaemonThreadManager::instance().removeThread(this);

    mutex.lock();

    TDevice *device;
    while ((device = (TDevice*)queue_remove(&events))) {
        delete device;
    }
    queue_clear(&events);

    mutex.unlock();

    LIST_FOR(&channelHeaders) {
        CSV_Channel_Header *ch = LIST_CUR(CSV_Channel_Header);
        delete[] ch->name;
        delete[] ch->id;
        delete[] ch->description;
        delete ch;
    }

    list_clear(&channelHeaders);

    delete table;
    delete[] datalog.comment;

    LIST_FOR(&datalog.devices) {
        Device *device = LIST_CUR(Device);
        device_delete(device);
    }

    list_clear(&datalog.devices);
    list_clear(&datalog.channels);

}

void LoggingThread::run() {

    QByteArray pathByteArray = path.toLocal8Bit();

    csvopt.file = fopen(pathByteArray.data(), fileMode);
    if (!csvopt.file) {
        emit error(ERROR_OPEN_OUTPUT);
        return;
    }

    datalog.creation_timestamp = timestamp_now();

    csv = csv_init(&csvopt);
    csv_write_header_custom(csv, &channelHeaders);

    while (processEvents());

    csv_exit(csv);
    fclose(csvopt.file);

    // Disconnect signals

    TDeviceManager& dm = TDeviceManager::instance();
    TDeviceSignal *signal = dm.getDeviceSignal(TDevice::TICKER_SERIAL_NUMBER);
    signal->disconnect(this);

    QMapIterator<QString, TDeviceSignal*> it(deviceSignalBySerial);

    while (it.hasNext()) {
        it.next();
        signal = it.value();
        signal->disconnect(this);
    }

}

void LoggingThread::shutdown() {

    mutex.lock();
    state = SHUTDOWN;
    condition.wakeOne();
    mutex.unlock();

}

bool LoggingThread::processEvents() {

    State currentState;
    TDevice *device;

    {
        QMutexLocker locker(&mutex);

        while (queue_size(&events) == 0 && state != SHUTDOWN) {
            condition.wait(&mutex);
        }

        currentState = state;
        device = (TDevice*) queue_remove(&events);

    }

    if (device) {

        QVector<TChannel*>& channels = device->channels;
        int64_t timestamp = device->timestamp;

        switch (device->event) {

            case TDevice::UPDATE:
                for (int i = 0; i < channels.size(); i++) {
                    table->insert(timestamp, channels[i]->id, channels[i]->calibratedQuantity);
                }
                break;

            case TDevice::CONNECT:
                for (int i = 0; i < channels.size(); i++) {
                    table->connect(timestamp, channels[i]->id);
                }
                break;

            case TDevice::DISCONNECT:
                for (int i = 0; i < channels.size(); i++) {
                    table->disconnect(timestamp, channels[i]->id);
                }
                break;

        }

        delete device;

    }

    writeRows();

    return currentState != SHUTDOWN;

}

void LoggingThread::writeRows() {

    if (!table->hasCompleteRow()) {
        return;
    }

    unsigned int n = 0;

    do {

        const DataRow *row = table->nextCompleteRow();

        datalog.timestamp = row->getTimestamp();

        for (int c = 0; c < channelCount; c++) {
            Channel *channel = LIST_GET(&datalog.channels, c, Channel);
            channel->quantity = row->getQuantity(c);
        }

        csv_write_row(csv);
        n++;

    } while (table->hasCompleteRow());

    fflush(csvopt.file);

    lineCounter += n;
    emit written(lineCounter);

}

void LoggingThread::onDeviceConnected(const TDevice *inputDevice) {

    addEvent(inputDevice);

}

void LoggingThread::onDeviceDisconnected(const TDevice *inputDevice) {

    addEvent(inputDevice);

}

void LoggingThread::onDeviceUpdated(const TDevice *inputDevice) {

    int64_t dt = inputDevice->timestamp - origin;

    if (dt % interval_us == 0) {
        addEvent(inputDevice);
    }

}

void LoggingThread::addEvent(const TDevice *device) {

    TDevice *copy = new TDevice(*device);

    mutex.lock();
    queue_add(&events, copy);
    condition.wakeOne();
    mutex.unlock();

}
