#include "Config.h"

#include <QStandardPaths>
#include <QDir>
#include <QFile>
#include <QJsonObject>
#include <QJsonDocument>
#include <QJsonValue>
#include <QDebug>

bool createDefaultConfigIfMissing() {
    QString roamingPath = QDir::fromNativeSeparators(qgetenv("APPDATA"));
	QString dracalPath = roamingPath + "/Dracal";
    QString configFilePath = dracalPath + "/dracal-view-config.json";

    QDir dracalDir(dracalPath);
    if (!dracalDir.exists()) {
        dracalDir.mkpath(".");
    }

    QFile configFile(configFilePath);
    if (configFile.exists()) {
        qDebug() << "default config file already exists at:" << configFilePath;
        return true;
    }

    QJsonObject defaultConfig;
    defaultConfig["local_tenkinet_port"] = "10395";
	defaultConfig["dracal_usb_service_port"] = "11395";
    QJsonDocument doc(defaultConfig);

    if (!configFile.open(QIODevice::WriteOnly)) {
        qWarning() << "Failed to create config file at:" << configFilePath;
        return false;
    }

    configFile.write(doc.toJson());
    configFile.close();
    qDebug() << "Created default config file at:" << configFilePath;
    return true;
}

QString loadTenkiNetPort() {
	static const auto port = []() -> QString {
        const QString default_port = "10395";
    
        if (!createDefaultConfigIfMissing()) {
            return default_port;
        }

		QString roamingPath = QDir::fromNativeSeparators(qgetenv("APPDATA"));
		QString dracalPath = roamingPath + "/Dracal";
		QString configFilePath = dracalPath + "/dracal-view-config.json";

		QFile configFile(configFilePath);
		if (!configFile.open(QIODevice::ReadOnly)) {
			qWarning() << "Failed to read config file:" << configFilePath;
			return default_port;
		}

		QByteArray data = configFile.readAll();
		configFile.close();

		QJsonDocument doc = QJsonDocument::fromJson(data);
		if (doc.isNull() || !doc.isObject()) {
			qWarning() << "Invalid JSON in config file";
			return default_port;
		}

		QJsonObject config = doc.object();
		if (!config.contains("local_tenkinet_port")) {
			qWarning() << "local_tenkinet_port not found in config file";
			return default_port;
		}

		QJsonValue portValue = config["local_tenkinet_port"];
		if (!portValue.isString()) {
			qWarning() << "local_tenkinet_port is not a string in config file";
			return default_port;
		}

		const auto port = portValue.toString();
		qDebug() << "Loaded local tenkinet port: " << port;
		return port;
	}();

	return port;
}

QString loadDracalUsbServicePort() {
	static const auto port = []() -> QString {
        const QString default_port = "11395";

        if (!createDefaultConfigIfMissing()) {
            return default_port;
        }

		QString roamingPath = QDir::fromNativeSeparators(qgetenv("APPDATA"));
		QString dracalPath = roamingPath + "/Dracal";
		QString configFilePath = dracalPath + "/dracal-view-config.json";

		QFile configFile(configFilePath);
		if (!configFile.open(QIODevice::ReadOnly)) {
			qWarning() << "Failed to read config file:" << configFilePath;
			return default_port;
		}

		QByteArray data = configFile.readAll();
		configFile.close();

		QJsonDocument doc = QJsonDocument::fromJson(data);
		if (doc.isNull() || !doc.isObject()) {
			qWarning() << "Invalid JSON in config file";
			return default_port;
		}

		QJsonObject config = doc.object();
		if (!config.contains("dracal_usb_service_port")) {
			qWarning() << "dracal_usb_service_port not found in config file";
			return default_port;
		}

		QJsonValue portValue = config["dracal_usb_service_port"];
		if (!portValue.isString()) {
			qWarning() << "dracal_usb_service_port is not a string in config file";
			return default_port;
		}

		const auto port = portValue.toString();
		qDebug() << "Loaded dracal usb service port: " << port;
		return port;
	}();

	return port;
}
