#include <QDoubleValidator>
#include <QGridLayout>
#include <QMessageBox>
#include <QDebug>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QJsonObject>
#include <QJsonDocument>
#include <QEventLoop>
#include <QTimer>

#include "Config.h"
#include "CalibrationEditDialog_DXC120.h"
#include "TUSBProvider.h"


#define FRESH_AIR_VALUE_MIN      400
#define FRESH_AIR_VALUE_DEFAULT  450
#define FRESH_AIR_VALUE_MAX     2000


CalibrationEditDialog_DXC120::CalibrationEditDialog_DXC120(QWidget *parent, const QString& serialNumber, const bool localTenkinet) :
	QDialog(parent),
	serialNumber(serialNumber), localTenkinet(localTenkinet)
{
	QDoubleValidator *valueValidator;

	btn_close = new QPushButton(tr("Close"));

	connect(btn_close, SIGNAL(clicked()), this, SLOT(close()));
	setWindowTitle(tr("Calibrate channel"));

	QGridLayout *gridLayout = new QGridLayout();

	QLabel *freshAirLabel = new QLabel("Fresh air (ppm)");
	freshAirEdit = new QLineEdit();
	valueValidator = new QDoubleValidator(this);
	valueValidator->setRange(FRESH_AIR_VALUE_MIN, FRESH_AIR_VALUE_MAX, 0);
	freshAirEdit->setValidator(valueValidator);
	btn_set_fresh_air = new QPushButton("Set fresh air");
	connect(btn_set_fresh_air, &QPushButton::clicked, this, &CalibrationEditDialog_DXC120::setFreshAir);

	dotsTimer = new QTimer(this);
	dotsTimer->setSingleShot(false);
	dotsTimer->setInterval(500);
	dotCount = 0;
	originalOkText = btn_set_fresh_air->text();

	connect(dotsTimer, &QTimer::timeout, [=]() {
		dotCount++;
		if (dotCount > 3) {
			dotCount = 1;
		}
		QString dots = QString(".").repeated(dotCount);
		btn_set_fresh_air->setText(dots);
	});

	gridLayout->addWidget(freshAirLabel, 0, 0);
	gridLayout->addWidget(freshAirEdit, 0, 1);
	gridLayout->addWidget(btn_set_fresh_air, 0, 2);

	cb_disable = new QCheckBox("Disable auto-calibration");

	connect(cb_disable, &QCheckBox::stateChanged, this, &CalibrationEditDialog_DXC120::autoCalibrationStateChanged);

	QHBoxLayout *lay_btns = new QHBoxLayout();
	lay_btns->addStretch();
	lay_btns->addWidget(btn_close);

	QVBoxLayout *lay_vert = new QVBoxLayout();
	lay_vert->addSpacing(10);
	lay_vert->addWidget(cb_disable);
	lay_vert->addSpacing(10);
	lay_vert->addStretch();
	lay_vert->addLayout(gridLayout);
	lay_vert->addLayout(lay_btns);

	setLayout(lay_vert);
	setModal(true);

	refresh();
}

CalibrationEditDialog_DXC120::~CalibrationEditDialog_DXC120()
{

}

void CalibrationEditDialog_DXC120::refresh()
{

	freshAirEdit->setText(QString::number(FRESH_AIR_VALUE_DEFAULT));

	int ascEnabled = -1;

	if (localTenkinet) {
		QNetworkAccessManager manager;
		const QString port = loadDracalUsbServicePort();
		const QString url = QString("http://localhost:%1/expert/3.4.1/get_dxc120_asc").arg(port);
		QNetworkRequest request{QUrl(url)};
		request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
		
		QJsonObject json;
		json["serial"] = serialNumber;

		// Convert to QByteArray for POST request
		QJsonDocument doc(json);
		QByteArray data = doc.toJson();

		QNetworkReply *reply = manager.post(request, data);
		
		// Block until finished
		QEventLoop loop;
		connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
		loop.exec();
		
		bool success = (reply->error() == QNetworkReply::NoError);
		int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

		if (success && statusCode == 200) {
			QByteArray responseData = reply->readAll();
			QJsonDocument responseDoc = QJsonDocument::fromJson(responseData);
			
			if (responseDoc.isObject()) {
				QJsonObject responseObj = responseDoc.object();
				if (responseObj.contains("value")) {
					ascEnabled = responseObj["value"].toInt();
				}
			}
		}

		reply->deleteLater();

	} else {
		ascEnabled = TUSBProvider::instance().get_DXC120_ASC(serialNumber);
	}

	if (ascEnabled < 0) {
		// Could not get ASC from sensor...
		cb_disable->setEnabled(false);
		freshAirEdit->setEnabled(false);
		btn_set_fresh_air->setEnabled(false);
	}
	else {
		cb_disable->setChecked(!ascEnabled);
		cb_disable->setEnabled(true);
		freshAirEdit->setEnabled(true);
		btn_set_fresh_air->setEnabled(true);
	}

}

void CalibrationEditDialog_DXC120::autoCalibrationStateChanged(int state)
{
	// Checked:   auto-calibration disabled
	// Unchecked: auto-calibration enabled

	int enabled = (state == Qt::Unchecked) ? 1 : 0;
	
	if (localTenkinet) {
		QNetworkAccessManager manager;
		const QString port = loadDracalUsbServicePort();
		const QString url = QString("http://localhost:%1/expert/3.4.1/set_dxc120_asc").arg(port);
		QNetworkRequest request{QUrl(url)};
		request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
		
		QJsonObject json;
		json["serial"] = serialNumber;
		json["value"] = enabled;

		// Convert to QByteArray for POST request
		QJsonDocument doc(json);
		QByteArray data = doc.toJson();

		QNetworkReply *reply = manager.post(request, data);
		
		// Block until finished
		QEventLoop loop;
		connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
		loop.exec();
		
		bool success = (reply->error() == QNetworkReply::NoError);
		int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
		
		reply->deleteLater();
	} else {
		TUSBProvider::instance().set_DXC120_ASC(serialNumber, enabled);
	}

}

void CalibrationEditDialog_DXC120::setFreshAir()
{
	if (freshAirEdit->hasAcceptableInput()) {
		QString value = freshAirEdit->text();
        if (localTenkinet) {
            btn_set_fresh_air->setEnabled(false);
            cb_disable->setEnabled(false);
            freshAirEdit->setEnabled(false);
            btn_close->setEnabled(false);
            dotCount = 1;
            btn_set_fresh_air->setText(".");
            dotsTimer->start();

            QNetworkAccessManager manager;
			const QString port = loadDracalUsbServicePort();
			const QString url = QString("http://localhost:%1/expert/3.4.1/set_dxc120_frc").arg(port);
			QNetworkRequest request{QUrl(url)};
			request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
			
			QJsonObject json;
			json["serial"] = serialNumber;
			json["value"] = value.toInt();

			// Convert to QByteArray for POST request
			QJsonDocument doc(json);
			QByteArray data = doc.toJson();

			QNetworkReply *reply = manager.post(request, data);
			
			// Block until finished
			QEventLoop loop;
			connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
			loop.exec();
			
			bool success = (reply->error() == QNetworkReply::NoError);
			int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
			
			reply->deleteLater();
            stopDots();
        } else {
			TUSBProvider::instance().set_DXC120_FRC(serialNumber, value.toInt());
		}
	}
	else {
		QMessageBox msgBox;
		QString message;
		message.sprintf("Value must be between %d and %d ppm", FRESH_AIR_VALUE_MIN, FRESH_AIR_VALUE_MAX);
		msgBox.setIcon(QMessageBox::Information);
		msgBox.setText(message);
		msgBox.exec();
	}
}

void CalibrationEditDialog_DXC120::stopDots()
{
	dotsTimer->stop();
	btn_set_fresh_air->setText(originalOkText);
	btn_set_fresh_air->setEnabled(true);
	cb_disable->setEnabled(true);
	freshAirEdit->setEnabled(true);
	btn_close->setEnabled(true);
}

void CalibrationEditDialog_DXC120::showEvent(QShowEvent *event) {

	refresh();
	QDialog::showEvent(event);

}
